/*
 *	Sherlock Indexer -- Getting Buckets from Indexer Sources
 *
 *	(c) 2003--2006 Martin Mares <mj@ucw.cz>
 *	(c) 2004--2005 Robert Spalek <robert@ucw.cz>
 */

#undef LOCAL_DEBUG

#include "sherlock/sherlock.h"
#include "lib/fastbuf.h"
#include "lib/ff-binary.h"
#include "sherlock/bucket.h"
#include "sherlock/object.h"
#include "sherlock/objread.h"
#include "sherlock/lizard-fb.h"
#include "lib/bbuf.h"
#include "indexer/indexer.h"

#include <fcntl.h>
#include <stdlib.h>
#include <pthread.h>

uns gb_max_count = ~0U;
static uns gb_count;
static uns gb_progress;
static uns gb_progress_max;
static struct fastbuf *gb_fb;
static uns gb_open_count;
static uns gb_threaded;
static struct get_buck *gb_master;
static struct obuck obuck;

struct gb_ops {
  int (*get)(struct get_buck *gb, uns oid);
  void (*init)(struct get_buck *gb);
  void (*cleanup)(struct get_buck *gb);
};

static void
gb_report_skip(struct get_buck *gb, uns oid)
{
  log(L_ERROR, "Bucket %x of type %x skipped: %m", gb->oid, gb->type);
  if (oid != ~0U)
    die("Fatal error: Bucket is unreadable, although it was readable before a while");
}

/*** Backward-Compatible Bucket Source ***/

static int
gb_old_get(struct get_buck *gb, uns oid)
{
  struct obuck_header bh;
  struct fastbuf *f;

  for(;;)
    {
      f = obuck_slurp_pool(&obuck, &bh, oid);
      if (!f)
	return 0;
      gb->type = bh.type;
      gb->oid = bh.oid;
      gb_progress = bh.oid;
      gb->o = obj_read_bucket(gb->buck_buf, gb->pool, gb->type, bh.length, f, NULL, !gb_threaded);
      if (likely(gb->o != NULL))
	return 1;
      gb_report_skip(gb, oid);
    }
}

static void
gb_old_cleanup(struct get_buck *gb UNUSED)
{
  obuck_cleanup(&obuck);
}

static void
gb_old_init(struct get_buck *gb)
{
  obuck_init(&obuck, gb->name, 0);
  gb_progress_max = obuck_predict_last_oid(&obuck);
}

static const struct gb_ops gb_old_ops = {
  .init = gb_old_init,
  .cleanup = gb_old_cleanup,
  .get = gb_old_get
};

/*** Text Source ***/

static int
gb_text_get_line(void)
{
  byte *buf = bgets_stk(gb_fb);
  if (!buf)
    return 0;
  if (!buf[0])
    gb_progress++;
  return 1;
}

static int
gb_text_get(struct get_buck *gb, uns oid)
{
  gb_progress++;
  if (oid != ~0U)
    while (gb_progress != oid)
      if (!gb_text_get_line())
	return 0;
  gb->type = BUCKET_TYPE_PLAIN;
  gb->oid = gb_progress;
  gb->o = obj_new(gb->pool);
  return obj_read(gb_fb, gb->o);
}

static void
gb_text_cleanup(struct get_buck *gb UNUSED)
{
  bclose(gb_fb);
}

static void
gb_text_init(struct get_buck *gb)
{
  gb_fb = bopen(gb->name, O_RDONLY, indexer_fb_size);
  gb_progress_max = ~0U;
}

static const struct gb_ops gb_text_ops = {
  .init = gb_text_init,
  .cleanup = gb_text_cleanup,
  .get = gb_text_get
};

/*** Raw bucket source ***/

static uns
bskip_align(struct fastbuf *fb)
{
  uns align = (1 << CARD_POS_SHIFT) - 1;
  sh_off_t pos = btell(fb);
  uns skip = pos & align;
  if (skip)
    bskip(fb, align+1-skip);
  return (pos + align) >> CARD_POS_SHIFT;
}

static int
gb_raw_get(struct get_buck *gb, uns oid)
{
  uns card_id = bskip_align(gb_fb);
  gb_progress++;
  if (oid != ~0U)
    while (card_id != oid)
      {
	uns len = bgetl(gb_fb);
	if (len == ~0U)
	  return 0;
	bskip(gb_fb, len);
	gb_progress++;
	card_id = bskip_align(gb_fb);
      }
  uns len = bgetl(gb_fb);
  gb->type = bgetc(gb_fb) + BUCKET_TYPE_PLAIN;
  gb->oid = card_id;
  uns sh = LIZARD_COMPRESS_HEADER - 1;
  gb->o = obj_read_bucket(gb->buck_buf, gb->pool, gb->type, len-sh, gb_fb, NULL, !gb_threaded);
  return !!gb->o;
}

static void
gb_raw_cleanup(struct get_buck *gb UNUSED)
{
  bclose(gb_fb);
}

static void
gb_raw_init(struct get_buck *gb)
{
  gb_fb = bopen(gb->name, O_RDONLY, indexer_fb_size);
  gb_progress_max = ~0U;
}

static const struct gb_ops gb_raw_ops = {
  .init = gb_raw_init,
  .cleanup = gb_raw_cleanup,
  .get = gb_raw_get
};

/*** Multiplexer ***/

static pthread_mutex_t gb_mutex;

static void
gb_connect(struct get_buck *gb)
{
  byte *w[4];
  int e = sepsplit(fn_source, ':', w, ARRAY_SIZE(w));
  if (e <= 0)
    die("Indexer.Source: Invalid syntax");
  gb->name = w[1];
  gb->aux = w[2];

  const struct gb_ops *ops;
  if (!strcmp(w[0], "bucket") && e == 2)
    ops = &gb_old_ops;
  else if (!strcmp(w[0], "text") && e == 2)
    ops = &gb_text_ops;
  else if (!strcmp(w[0], "raw") && e == 2)
    ops = &gb_raw_ops;
  else
    die("Indexer.Source: Unknown source type");

  gb->ops = ops;
  ops->init(gb);
  pthread_mutex_init(&gb_mutex, NULL);

  gb_progress = 0;
  gb_count = 0;
}

void
get_buck_init(struct get_buck *gb)
{
  if (!gb_open_count++)
    {
      gb_connect(gb);
      gb_master = gb;
    }
  if (gb_open_count > 1)
    gb_threaded = 1;

  gb->ops = gb_master->ops;
  gb->buck_buf = buck2obj_alloc();
  gb->progress_max = gb_progress_max;
}

int
get_buck_next(struct get_buck *gb, uns oid)
{
  if (gb_threaded)
    pthread_mutex_lock(&gb_mutex);

  int ok = 0;
  if (gb_count < gb_max_count && gb->ops->get(gb, oid))
    {
      gb->progress_current = gb_progress;
      gb->progress_count = ++gb_count;
      ok = 1;
    }

  if (gb_threaded)
    pthread_mutex_unlock(&gb_mutex);
  return ok;
}

void
get_buck_cleanup(struct get_buck *gb)
{
  ASSERT(gb_open_count);
  if (!--gb_open_count)
    gb->ops->cleanup(gb);
  buck2obj_free(gb->buck_buf);
}

#ifdef TEST

#include "lib/getopt.h"
#include "lib/mempool.h"

int main(int argc, char **argv)
{
  log_init(argv[0]);
  if (cf_getopt(argc, argv, CF_SHORT_OPTS, CF_NO_LONG_OPTS, NULL) >= 0)
    return 1;
  if (optind < argc)
    fn_source = argv[optind];
  log(L_INFO, "Getting buckets from %s", fn_source);

  struct mempool *mp = mp_new(16384);
  struct get_buck gb;
  get_buck_init(&gb);
  gb.pool = mp;
  struct fastbuf *out = bfdopen_shared(1, 65536);
  for (; get_buck_next(&gb, ~0U); mp_flush(mp))
    {
      bprintf(out, "# %08x (#%d; %d of %d), type %08x\n", gb.oid, gb.progress_count, gb.progress_current, gb.progress_max, gb.type);
      obj_write(out, gb.o, BUCKET_TYPE_PLAIN);
      bputc(out, '\n');
    }
  bclose(out);
  get_buck_cleanup(&gb);
  return 0;
}

#endif
