/*
 *	Measure Timing of Random File Accesses
 *
 *	(c) 2001 Martin Mares <mj@ucw.cz>
 */

#include "sherlock/sherlock.h"
#include "lib/lfs.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <fcntl.h>
#include <unistd.h>
#include <alloca.h>
#include <sys/mman.h>

int
main(int argc, char **argv)
{
  int attempts = 1000;
  u64 maxsize = 0;
  u64 size;
  int mm = 0;
  int align = 0;
  int chunk = 1;
  char *file;
  int c, fd, i;
  byte *map = NULL;
  byte *buf;
  uns tim, rsize;

  while ((c = getopt(argc, argv, "mpn:s:a:c:")) >= 0)
    switch (c)
      {
      case 'n':	attempts = atol(optarg); break;
      case 's':	maxsize = (u64)atol(optarg) << 20; break;
      case 'm':	mm=1; break;
      case 'p': mm=2; break;
      case 'a':	align = atol(optarg); break;
      case 'c':	chunk = atol(optarg); break;
      default:
	fprintf(stderr, "Invalid arguments; consult source for syntax.\n");
	return 1;
      }
  if (optind != argc - 1)
    {
      fprintf(stderr, "Invalid arguments; consult source for syntax.\n");
      return 1;
    }
  file = argv[optind];

  fd = sh_open(file, O_RDONLY, 0);
  if (fd < 0)
    die("open: %m");
  size = sh_seek(fd, 0, SEEK_END);
  if (maxsize && size > maxsize)
    size = maxsize;

  printf("Measuring random accesses to %s (%Ld bytes)\n", file, size);
  while ((1ULL << (30 + align)) < size)
    align++;
  rsize = (size-chunk) >> align;
  printf("Reading %d chunks of %d bytes aligned at %d bytes (there are %u of them)\n", attempts, chunk, 1<<align, rsize);
  switch (mm)
    {
    case 1:
      puts("Using mmap() to read the file.");
      map = sh_mmap(NULL, size, PROT_READ, MAP_SHARED, fd, 0);
      if (map == MAP_FAILED)
	die("mmap: %m");
      break;
    case 2:
      puts("Using pread() to read the file.");
      break;
    default:
      puts("Using seek() and read() to read the file.");
    }

  buf = alloca(chunk);
  init_timer();
  for (i=0; i<attempts; i++)
    {
      sh_off_t pos = (sh_off_t) random_max(rsize) << align;
      sh_off_t res;
      switch (mm)
	{
	case 1:
	  memcpy(buf, map + pos, chunk);
	  break;
	case 2:
	  res = sh_pread(fd, buf, chunk, pos);
	  ASSERT(res == chunk);
	  break;
	default:
	  res = sh_seek(fd, pos, SEEK_SET);
	  ASSERT(res >= 0);
	  res = read(fd, buf, chunk);
	  ASSERT(res == chunk);
	}
    }

  tim = get_timer();
  printf("It took %d ms -> %.3f ms per access\n", tim, (double)tim/attempts);
  return 0;
}
