/*
 *	Sherlock Indexer -- Getting Buckets from the Gatherer
 *
 *	(c) 2003--2004 Martin Mares <mj@ucw.cz>
 *	(c) 2004 Robert Spalek <robert@ucw.cz>
 */

#undef LOCAL_DEBUG

#include "sherlock/sherlock.h"
#include "lib/fastbuf.h"
#include "sherlock/bucket.h"
#include "sherlock/object.h"
#include "lib/bbuf.h"
#include "indexer/indexer.h"

#include <fcntl.h>
#include <stdlib.h>

static struct bucket_source bucket_source;

static void
gb_report_skip(struct bucket_source *src, u32 oid)
{
  log(L_ERROR, "Bucket %x of type %x skipped: %m", src->oid, src->type);
  if (oid != ~0U)
    die("Fatal error: Bucket is unreadable, although it was readable before a while");
}

/*** Backward-Compatible Bucket Source ***/

static int
gb_old_get_next(struct bucket_source *src, struct mempool *mp, u32 oid)
{
  struct obuck_header bh;
  struct fastbuf *f;

  for(;;)
    {
      f = obuck_slurp_pool(&bh, oid);
      if (!f)
	return 0;
      src->type = bh.type;
      src->oid = bh.oid;
      src->progress_current = bh.oid;
      src->o = obj_read_bucket(src->buck_buf, mp, src->type, bh.length, f, NULL);
      if (likely(src->o != NULL))
	return 1;
      gb_report_skip(src, oid);
    }
}

static void
gb_old_cleanup(struct bucket_source *src)
{
  obuck_cleanup();
  buck2obj_free(src->buck_buf);
}

static void
gb_old_init(byte *bfname)
{
  obuck_name = bfname;
  obuck_init(0);
  bucket_source.get_next = gb_old_get_next;
  bucket_source.cleanup = gb_old_cleanup;
  bucket_source.progress_max = obuck_predict_last_oid();
}

/*** Text Source ***/

static int
gb_text_get_next(struct bucket_source *src, struct mempool *mp, u32 oid)
{
  byte buf[MAX_ATTR_SIZE];

  src->progress_current++;
  if (oid != ~0U)
    while (src->progress_current != oid)
      {
	if (!bgets(src->in_file, buf, sizeof(buf)))
	  return 0;
	if (!buf[0])
	  src->progress_current++;
      }
  src->type = BUCKET_TYPE_PLAIN;
  src->oid = src->progress_current;
  src->o = obj_new(mp);
  int have_attr = 0;
  while (bgets(src->in_file, buf, sizeof(buf)) && buf[0])
    {
      obj_add_attr(src->o, buf[0], buf+1);
      have_attr = 1;
    }
  return have_attr;
}

static void
gb_text_cleanup(struct bucket_source *src)
{
  bclose(src->in_file);
}

static void
gb_text_init(byte *fname)
{
  bucket_source.in_file = bopen(fname, O_RDONLY, indexer_fb_size);
  bucket_source.get_next = gb_text_get_next;
  bucket_source.cleanup = gb_text_cleanup;
  bucket_source.progress_max = ~0U;
}

/*** Multiplexer ***/

struct bucket_source *
get_buck_init(void)
{
  bzero(&bucket_source, sizeof(bucket_source));
  bucket_source.buck_buf = buck2obj_alloc();

  byte *w[4];
  int e = sepsplit(fn_source, ':', w, ARRAY_SIZE(w));
  if (e <= 0)
    die("Indexer.Source: Invalid syntax");

  if (!strcmp(w[0], "bucket") && e == 2)
    gb_old_init(w[1]);
  else if (!strcmp(w[0], "text") && e == 2)
    gb_text_init(w[1]);
  else
    die("Indexer.Source: Unknown source type");

  return &bucket_source;
}

#ifdef TEST

#include "lib/conf.h"
#include "lib/mempool.h"

int main(int argc, char **argv)
{
  log_init(argv[0]);
  if (cf_getopt(argc, argv, CF_SHORT_OPTS, CF_NO_LONG_OPTS, NULL) >= 0)
    return 1;
  if (optind > argc)
    fn_source = argv[optind];

  struct mempool *mp = mp_new(16384);
  struct bucket_source *src = get_buck_init();
  struct fastbuf *out = bfdopen_shared(1, 65536);
  for (; src->get_next(src, mp, ~0U); mp_flush(mp))
    {
      bprintf(out, "# %08x (%d of %d), type %08x\n", src->oid, src->progress_current, src->progress_max, src->type);
      obj_write(out, src->o);
      bputc(out, '\n');
    }
  bclose(out);
  src->cleanup(src);
  return 0;
}

#endif
