/*
 *	Sherlock Language Processing Library -- Stemming
 *
 *	(c) 2003--2004 Martin Mares <mj@ucw.cz>
 */

#include "sherlock/sherlock.h"
#include "sherlock/index.h"
#include "lib/mempool.h"
#include "lang/lang.h"
#include "lang/stemmer.h"

#include <string.h>

static list *(*stemmer_funcs[])(struct stemmer *, byte *, struct mempool *) = {
#define STEMMER(name) stem_##name,
#include "lang/stemmers.h"
#undef STEMMER
};

static list *(*expander_funcs[])(struct stemmer *, byte *, struct mempool *) = {
#define STEMMER(name) stem_expand_##name,
#include "lang/stemmers.h"
#undef STEMMER
};

static void (*stemmer_init_funcs[])(struct stemmer *) = {
#define STEMMER(name) stem_init_##name,
#include "lang/stemmers.h"
#undef STEMMER
};

list *
lang_stem(struct stemmer *st, byte *src, struct mempool *mp)
{
  ASSERT(st->id < ARRAY_SIZE(stemmer_funcs));
  return stemmer_funcs[st->id](st, src, mp);
}

list *
lang_expand(struct stemmer *st, byte *src, struct mempool *mp)
{
  ASSERT(st->id < ARRAY_SIZE(stemmer_funcs));
  return expander_funcs[st->id](st, src, mp);
}

void
lang_init_stemmers(void)
{
  struct stemmer *st;
  WALK_LIST(st, stemmer_list)
    stemmer_init_funcs[st->id](st);
}

struct word_node *
word_list_add(struct mempool *mp, list *l, byte *w)
{
  int len = strlen(w);
  if (len >= MAX_WORD_LEN)
    return NULL;
  struct word_node *n = mp_alloc(mp, sizeof(*n) + len);
  memcpy(n->w, w, len+1);
  add_tail(l, &n->n);
  return n;
}

struct word_node *
word_list_find(list *l, byte *w)
{
  struct word_node *n;
  WALK_LIST(n, *l)
    if (!strcmp(n->w, w))
      return n;
  return NULL;
}

struct word_node *
word_list_add_unique(struct mempool *mp, list *l, byte *w)
{
  struct word_node *n = word_list_find(l, w);
  if (n)
    return n;
  else
    return word_list_add(mp, l, w);
}
