/*
 *	Sherlock Search Engine -- Card Files
 *
 *	(c) 1997--2005 Martin Mares <mj@ucw.cz>
 *	(c) 2001--2004 Robert Spalek <robert@ucw.cz>
 */

#undef LOCAL_DEBUG

#include "sherlock/sherlock.h"
#include "lib/mempool.h"
#include "lib/hashfunc.h"
#include "lib/heap.h"
#include "lib/url.h"
#include "sherlock/index.h"
#include "sherlock/tagged-text.h"
#include "lib/unicode.h"
#include "sherlock/lizard-fb.h"
#include "sherlock/object.h"
#include "indexer/lexicon.h"
#include "search/sherlockd.h"
#include "charset/unicat.h"

#ifdef CONFIG_LANG
#include "lang/lang.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <alloca.h>

#define	TRACE(mask...)	do { if (q->debug & DEBUG_DUMPING) log(L_DEBUG, mask); } while (0)

int
check_result_set(struct query *q)
{
  struct val_set *rng;

  if (!q->range)
    {
      rng = q->range = mp_alloc_zero(q->pool, sizeof(*rng));
      rng->min = 1;
      rng->max = num_matches;
    }
  for (rng=q->range; rng; rng=rng->next)
    if (rng->max > num_matches)
      {
	add_qerr("-106 Too many documents requested, only first %d matches available.", num_matches);
	return 1;
      }
    else if (rng->min < 1)
      {
	add_qerr("-106 Matches are numbered starting with 1");
	return 1;
      }
  return 0;
}

static void
show_card_header(struct query *q, struct database *db, oid_t oid, struct result_note *note)
{
  struct card_attr *ca = note->attr;

  reply_f("B%s", db->name);
  reply_f("O%08x", oid);
  reply_f("Q%d", note->q);
  reply_f("w%d", ca->weight);
#ifdef CONFIG_LANG
  if (CA_GET_FILE_LANG(ca))
    reply_f("l%s", lang_code_to_name(CA_GET_FILE_LANG(ca)));
#endif
  CUSTOM_MATCH_SHOW(q, ca, reply_f);

  if (q->debug & DEBUG_CARD_INFO)
    {
      int i;
      reply_f(".K%08x", note->sec_sort_key);
      for (i=0; i<HARD_MAX_NOTES; i++)
	if (note->best[i] != 0xffff)
	  reply_f(".R%04x", note->best[i]);
#ifdef CONFIG_LASTMOD
      reply_f(".Cage=%d", ca->age);
#endif
#define INT_ATTR(id,kw,gf,pf) reply_f(".C" #id "=%d", gf(ca));
#define SMALL_SET_ATTR INT_ATTR
#define LATE_INT_ATTR INT_ATTR
#define LATE_SMALL_SET_ATTR SMALL_SET_ATTR
      EXTENDED_ATTRS
#undef SMALL_SET_ATTR
#undef INT_ATTR
#undef LATE_INT_ATTR
#undef LATE_SMALL_SET_ATTR
    }
}

static void
show_results_list(struct query *q)
{
  uns from, to;
  struct results *res = q->results;
  struct val_set *rng;

  profiler_switch(&prof_results);
  for (rng=q->range; rng; rng=rng->next)
    {
      from = rng->min;
      to = MIN(rng->max, res->nresults);
      while (from <= to)
	{
	  struct result_note *n = res->result_heap[from++];
	  oid_t oid;
	  struct database *db = attr_to_db(n->attr, &oid);
	  show_card_header(q, db, oid, n);
	  reply_f("%s", "");		/* Avoid format string warning */
	}
    }
}

/***** Lexical Mapping *****/

#define	UNSET_POS	0xffff
#define	UNSET_TYPE	0xff
struct lexmap_word
{
	struct lexmap_word *next;
	byte *orig;		/* pointer to the original text */
	word pos;		/* position */
	byte olen;		/* and word length */
	byte type;		/* word type */
};
typedef struct lexmap_word *word_id_t;

struct hilite_hash;
struct lexmap_list
{
	struct hilite_hash *hh;
	struct database *db;
	struct lexmap_word *first_word, **p_last_word;
	struct lexmap_word *curr_word;
	byte *text, *text_end;
};

static struct lexmap_list wptr;			/* storage space for the document text parsed into words */
static inline uns lexmap_eof(void) { return !wptr.curr_word->next; }
static inline void lexmap_next(void) { wptr.curr_word = wptr.curr_word->next; }
static inline void lexmap_first(void) { wptr.curr_word = wptr.first_word; }

static void
lexmap_find_word(uns limit)
{
  	while (!lexmap_eof() && wptr.curr_word->pos == UNSET_POS)
		lexmap_next();
	if (lexmap_eof() || wptr.curr_word->pos > limit)
		lexmap_first();
	while (!lexmap_eof() && (wptr.curr_word->pos == UNSET_POS || wptr.curr_word->pos < limit))
		lexmap_next();
}

static void
lexmap_find_char(byte *limit)
{
	if (lexmap_eof() || wptr.curr_word->orig > limit)
		lexmap_first();
	while (!lexmap_eof() && wptr.curr_word->orig < limit)
		lexmap_next();
}

static struct mempool *lm_pool;			/* memory pool for the parsing operation */

static enum word_class
lm_lookup(enum word_class orig_class, word *uni, uns ulen, word_id_t *p, byte *orig, uns olen)
{
	struct lexmap_word *idp;
	byte wbuf[2*MAX_WORD_LEN+1], *wp=wbuf;
	uns wl, i;
	enum word_class wc;

	if (!uni)
		return orig_class;
	for (i=0; i<ulen; i++)
	{
		uns u = *uni++;
		u = Utolower(u);
		PUT_UTF8(wp, u);
	}
	*wp = 0;
	wl = wp - wbuf + 1;
	idp = mp_alloc(lm_pool, sizeof(struct lexmap_word));
	idp->pos = UNSET_POS;
	idp->type = 0;
	idp->orig = orig;
	idp->olen = MIN(olen, 0xff);
	if (orig_class != WC_NORMAL)
		wc = orig_class;
	else
	{
		int c = word_classify(wptr.db, wbuf);
		if (c < 0)
			c = WC_NORMAL;
		wc = c;
	}
	*p = idp;
	*wptr.p_last_word = idp;
	wptr.p_last_word = &idp->next;
	return wc;
}

static inline void
lexmap_set_pos(struct lexmap_word *p, uns pos)
{
	if (pos < UNSET_POS)
	{
		if (p->pos == UNSET_POS)
			p->pos = pos;
		else
			ASSERT(p->pos == pos);
	}
}

static void
lm_got_word(uns pos, uns type, word_id_t p)
{
	lexmap_set_pos(p, pos);
	p->type = type;
}

static void
lm_got_complex(uns pos, uns type, word_id_t root, word_id_t context, uns dir)
{
	lexmap_set_pos(root, pos);
	root->type = type;
	lexmap_set_pos(context, dir ? pos+1 : pos-1);
}

#define LM_TRACK_TEXT
#define LM_CARDS
#include "indexer/lexmap.h"

static void
lexmap_parse(byte *text, byte *text_end, uns starting_pos, uns parse_url)
{
	if (!lm_pool)
		lm_pool = mp_new(16384);
	else
		mp_flush(lm_pool);
	wptr.text = text;
	wptr.text_end = text_end;

	lm_doc_start();
	lm_pos = starting_pos;
	wptr.p_last_word = &wptr.first_word;
	if (parse_url)
		lm_map_url(text, text_end);
	else
		lm_map_text(text, text_end);

	struct lexmap_word *tail = mp_alloc(lm_pool, sizeof(struct lexmap_word));
	tail->next = NULL;
	tail->pos = UNSET_POS;
	tail->orig = text_end;
	tail->olen = 0;
	tail->type = UNSET_TYPE;
	*wptr.p_last_word = tail;
	lexmap_first();
	/* There might remain some words in the list with pos==UNSET_POS;
	 * lm_got_{word,context}() has not been called for them.  For example,
	 * context words without a context (surrounded by two breaks) or
	 * ignored words.  Let their position remain unset.  */
}

/***** Hash-table of highlighted words  *****/

#define	HILITE_HASH_SIZE	256

struct hilite_hash
{
	uns size;
	struct hilite_word *h[0];
};

static inline uns
hilhash_hashf(byte *w, byte *w_end)
{
	if (w_end)
		return hash_block(w, w_end - w);
	else
		return hash_string(w);
}

static struct hilite_hash *
hilhash_create(struct query *q, uns size, struct hilite_word *words)
{
	struct hilite_hash *hilhash = mp_alloc_zero(q->pool,
		sizeof(struct hilite_hash) + size*sizeof(struct hilite_word *));
	hilhash->size = size;
	for (uns i=0; words; words=words->next, i++)
	{
		uns j = hilhash_hashf(words->w, NULL) % hilhash->size;
		struct hilite_word *w = mp_alloc_zero(q->pool,
			sizeof(struct hilite_word) + strlen(words->w));
		strcpy(w->w, words->w);
		w->next = hilhash->h[j];
		hilhash->h[j] = w;
		TRACE("Hilite %d: %s", i, words->w);
	}
	return hilhash;
}

static inline struct hilite_word *
hilhash_search(struct hilite_hash *hilite, byte *w, byte *w_end)
{
	uns j;
	struct hilite_word *hw;
	j = hilhash_hashf(w, w_end) % hilite->size;
	hw = hilite->h[j];
	while (hw && strcmp(w, hw->w))
		hw = hw->next;
	return hw;
}

static uns
word_tolower_utf8(byte *w, byte *w_end, byte *to)
{
	byte *buf = to;
	while (w < w_end)
	{
		uns u;
		GET_UTF8(w, u);
		if (u < 0x80000000 && alpha_class[u] == AC_LIGATURE)
		{
			const word *lig = Uexpand_lig(u);
			for (uns i=0; lig[i]; i++)
			{
				uns c = Utolower(lig[i]);
				PUT_UTF8(buf, c);
			}
		}
		else
		{
			u = Utolower(u);
			PUT_UTF8(buf, u);
		}
	}
	*buf = 0;
	return buf - to;
}

static int
hilhash_found(void)
{
	byte w[2*MAX_WORD_LEN+1];
	word_tolower_utf8(wptr.curr_word->orig, wptr.curr_word->orig + wptr.curr_word->olen, w);
	return !! hilhash_search(wptr.hh, w, NULL);
}

static uns
hilhash_count_found_words(void)
{
	uns count = 0;
	for (lexmap_first(); !lexmap_eof(); lexmap_next())
	{
		if (hilhash_found())
			count++;
	}
	return count;
}

/***** Parsing of result notes *****/

enum text_mode { TM_META, TM_WORD };				/* Different types are used for metas and words.  */
static byte tm_letter[2] = "MX";
byte *wt_names[8] = { WORD_TYPE_USER_NAMES };
byte *mt_names[16] = { META_TYPE_USER_NAMES };
byte *st_names[8] = { STRING_TYPE_USER_NAMES };

#define	RESNOTE_INVALID_POS	-1U
static inline uns
result_note_pos(u16 best, enum text_mode mode, uns want_type)
{
	if (mode == TM_WORD)
	{
		if (best & 0x8000)
			return RESNOTE_INVALID_POS;
		else
		{
			ASSERT(!(best & 0x7000));
			return best & 0x0fff;
		}
	}
	else
	{
		if (!(best & 0x8000))
			return RESNOTE_INVALID_POS;
		else
		{
			uns type = (best & 0x7800) >> 11;
			ASSERT(!(best & 0x0600));
			uns pos = best & 0x01ff;
			if (type != want_type)
				return RESNOTE_INVALID_POS;
			else
				return pos;
		}
	}
}

/***** Dumper of a block of text *****/

struct context_interval
{
	byte *first_char, *last_char;			/* pointers to text */
	uns start_type;					/* type in the beginning */
	int context_len;				/* length of context in printable characters */
};

#define	DUMP_INTERTAG_SPACES	0			/* whether to insert spaces between tags */

#define	MAXLINE	72					/* maximal length of output lines */
#define	DUMP_CONST(sp, txt)	dump_word(&line_len, txt, sizeof(txt)-1, sp, mode)
#define	DUMP_MASK(sp, mask...)	{ \
	byte tmpbuf[64]; \
	uns tmplen = sprintf(tmpbuf, mask); \
	dump_word(&line_len, tmpbuf, tmplen, sp, mode); \
}

static void
dump_word(uns *line_len, byte *start, uns strlen, uns space, enum text_mode mode)
{
	if (!strlen)
	{
		/* Empty string means flushing the buffer.  */
		if (*line_len)
			*line_len = 0, reply_string(current_query, "\n", 1);
	}
	else
	{
		/* The text is broken into lines only at a space in the word mode.  */
		if (mode == TM_WORD && space && *line_len >= MAXLINE)
			*line_len = 0, reply_string(current_query, "\n", 1);
		if (!*line_len)
			*line_len = 1, reply_string(current_query, &tm_letter[mode], 1);
		else if (space)
			(*line_len)++, reply_string(current_query, " ", 1);
		reply_string(current_query, start, strlen);
		*line_len += strlen;
	}
}

static inline byte *
html_escape(uns c)
{
	switch (c)
	{
		case '<':
			return "&lt;";
		case '>':
			return "&gt;";
		case '&':
			return "&amp;";
		default:
			return NULL;
	}
}

static void
dump_text_interval(uns *noted_pos, enum text_mode mode, struct context_interval *range)
{
	byte **type_names = (mode == TM_META ? mt_names : wt_names);
	byte *text;
	uns line_len;
	uns space;
	uns type;

	text = range->first_char;
	type = range->start_type;
	space = 1;
	line_len = 0;
	DUMP_MASK(1, "<block c=%d-%d l=%d>",
		range->first_char - wptr.text, range->last_char - wptr.text, range->context_len);
	DUMP_MASK(DUMP_INTERTAG_SPACES, "<%s>", type_names[type]);
	while (text < range->last_char)
	{
		uns c = 0;
		int empty = 1;
		byte *bow = text, *eow = text;
		while (text < range->last_char)
		{
			GET_TAGGED_CHAR(text, c);
			if (c >= 0x80000000
			|| (alpha_class[c] != AC_ALPHA
			  && alpha_class[c] != AC_DIGIT
			  && alpha_class[c] != AC_LIGATURE))	// assuming that the expansion of the ligature consists of only alphanumerical characters
				break;
			empty = 0;
			eow = text;
		}
		lexmap_find_char(bow);
			/* We can reach EOF here due to the existence of non-indexed words.  */
		if (!empty)
		{
			uns j;
			int sign = 0;
			if (wptr.curr_word->orig == bow)
			{
				for (j=0; noted_pos[j] != RESNOTE_INVALID_POS; j++)
					if (wptr.curr_word->pos == noted_pos[j])
					{
						sign = 2;
						break;
					}
				if (!sign)
				{
					if (hilhash_found())
						sign = 1;
				}
			}
			if (sign == 2)
				DUMP_CONST(space, "<best>");
			else if (sign == 1)
				DUMP_CONST(space, "<found>");
			if (sign)
				space = 0;
			dump_word(&line_len, bow, eow - bow, space, mode);
			if (sign == 2)
				DUMP_CONST(0, "</best>");
			else if (sign == 1)
				DUMP_CONST(0, "</found>");
			space = 0;
		}
		if (c >= 0x80000000)
		{
			if (c < 0x80010000)
			{
				uns new_type = c & 0x0f;
				if (new_type != type)
					DUMP_MASK(DUMP_INTERTAG_SPACES, "</%s>", type_names[type]);
				if (c & 0x10 && eow > range->first_char)
					DUMP_CONST(DUMP_INTERTAG_SPACES, "<break>");
				if (new_type != type)
					DUMP_MASK(DUMP_INTERTAG_SPACES, "<%s>", type_names[new_type]);
				type = new_type;
			}
			else
				ASSERT(0);
			space = 1;
		}
		else if (Uspace(c))
		{
			space = 1;
		}
		else if (eow < range->last_char)			/* slash, ... */
		{
			byte *printing = html_escape(c);
			if (printing)
				dump_word(&line_len, printing, strlen(printing), space, mode);
			else
			{
				byte b[7];
				printing = b;
				PUT_UTF8(printing, c);
				dump_word(&line_len, b, printing-b, space, mode);
			}
			space = 0;
		}
	}
	DUMP_MASK(1, "</%s>", type_names[type]);
	DUMP_CONST(DUMP_INTERTAG_SPACES, "</block>");
	dump_word(&line_len, NULL, 0, 0, mode);
}

/***** Computing intervals that will be dumped *****/

#define	IS_SENTENCE_BREAK(c)	((c) == '.' || (c) == '?' || (c) == '!')

#define	PREV_TAGGED_CHAR(pos, pos1, limit, c) {\
	byte *pos2;\
	pos1 = pos;\
	do { pos1--; }\
	while (pos1 > limit && *pos1 >= 0x80 && *pos1 < 0xc0);\
	do { pos2 = pos1; GET_TAGGED_CHAR(pos1, c); }\
	while (pos1 < pos);\
	pos1 = pos2;\
}

static void
contint_delete(struct context_interval *ints, uns maxints, uns i)
{
	for (; i<maxints-1 && ints[i].first_char; i++)
		ints[i] = ints[i+1];
	bzero(ints + i, sizeof(struct context_interval));
}

static void
contint_insert(struct context_interval *ints, uns maxints, uns i)
{
	ASSERT(!ints[maxints-1].first_char);
	for (uns j=maxints-1; j>i; j--)
		ints[j] = ints[j-1];
	bzero(ints + i, sizeof(struct context_interval));
}

static uns
contint_add(struct query *q, byte *pos, byte *pos_end, struct context_interval *ints, uns maxints, int available)
{
	if (available <= 0)
		return 0;
	ASSERT(pos && pos_end && pos <= pos_end);
	ASSERT(pos >= wptr.text && pos_end <= wptr.text_end);
	TRACE("contint_add() entered with pos %d+%d and context %d",
		pos - wptr.text, pos_end - pos, available);

	/* Decide whether we hit an existing interval and compute the bounds
	 * for the stretching operation.  The interval array is sorted and the
	 * interval are pairwise disjunct and non-empty.  */
	struct context_interval tmp_int, *I = NULL;
	byte *softl = wptr.text;
	byte *softr = wptr.text_end;
	for (uns i = 0; i < maxints && ints[i].first_char; i++)
	{
		if (pos >= ints[i].first_char && pos < ints[i].last_char
		|| pos_end > ints[i].first_char && pos_end <= ints[i].last_char
		|| pos < ints[i].first_char && pos_end > ints[i].last_char)
		{
			I = ints + i;
			TRACE("Already contained by interval %d", i);
		}
		else if (ints[i].last_char <= pos)
		{
			ASSERT(ints[i].last_char > softl);
			softl = ints[i].last_char;
		}
		else if (ints[i].first_char >= pos_end)
		{
			softr = ints[i].first_char;
			break;
		}
	}
	uns total_added = 0;
	if (!I)
	{
		/* We might already have reached the maximum number of
		 * intervals, but ignore this for now, since the new interval
		 * might be merged during stretching.  */
		I = &tmp_int;
		I->first_char = pos;
		I->last_char = pos_end;
		I->start_type = UNSET_TYPE;
		I->context_len = pos_end - pos;
		TRACE("Temporarily created new interval of length %d", I->context_len);
		total_added = I->context_len;
	}
	else
	{
		/* Enlarge the original interval if needed.  */
		if (pos < I->first_char)
		{
			total_added += I->first_char - pos;
			I->context_len += I->first_char - pos;
			I->first_char = pos;
		}
		if (pos_end > I->last_char)
		{
			total_added += pos_end - I->last_char;
			I->context_len += pos_end - I->last_char;
			I->last_char = pos_end;
		}
	}
	available -= total_added;

	/* Try to stretch the current interval backward.  */
	int added = 0;
	int limit1 = available/3;
	int limit2 = available/2;
	uns c;
	pos = I->first_char;
	while (pos > softl)
	{
		byte *pos1;
		PREV_TAGGED_CHAR(pos, pos1, wptr.text, c);
		if (c < 0x8000000)
		{
			added++;
			if (added >= limit2 && Uspace(c)
			|| added >= limit1 && IS_SENTENCE_BREAK(c))
				break;
		}
		else if (c >= 0x80000000 && c < 0x80001000)
		{
			if (added >= limit1)
				break;
		}
		pos = pos1;
	}
	if (added)
		TRACE("Stretched to the left side by %d characters", added);
	available -= added;
	total_added += added;
	I->context_len += added;
	I->first_char = pos;
	/* Find out the new starting type.  */
	GET_TAGGED_CHAR(pos, c);
	while (1)
	{
		ASSERT(pos > wptr.text);	/* M's and X's should always start by changing the type */
		byte *pos1;
		PREV_TAGGED_CHAR(pos, pos1, wptr.text, c);
		if (c >= 0x80000000 && c < 0x80001000)
		{
			I->start_type = c & 0x0f;
			break;
		}
		pos = pos1;
	}

	/* Try to stretch the current interval forward.  */
	added = 0;
	limit1 = available*2/3;
	limit2 = available;
	pos = I->last_char;
	while (pos < softr)
	{
		byte *pos1;
		uns c;
		pos1 = pos;
		GET_TAGGED_CHAR(pos, c);
		if (c < 0x8000000)
		{
			added++;
			if (added >= limit2 && Uspace(c)
			|| added >= limit1 && IS_SENTENCE_BREAK(c))
			{
				if (Uspace(c))
					added--, pos = pos1;
				break;
			}
		}
		else if (c >= 0x80000000 && c < 0x80001000)
		{
			if (added >= limit1)
			{
				pos = pos1;
				break;
			}
		}
	}
	if (added)
		TRACE("Stretched to the right side by %d characters", added);
	available -= added;
	total_added += added;
	I->context_len += added;
	I->last_char = pos;

	if (I->first_char == softl && softl > wptr.text)
	{
		/* Merge to the left.  */
		uns i;
		for (i = 0; ints[i].last_char != I->first_char; i++);
		ints[i].last_char = I->last_char;
		ints[i].context_len += I->context_len;
		TRACE("Merged with interval %d on the left side", i);
		if (I != &tmp_int)
		{
			ASSERT(I == ints + (i+1));
			contint_delete(ints, maxints, i+1);
		}
		I = ints+i;
	}
	if (I->last_char == softr && softr < wptr.text_end)
	{
		/* Merge to the right.  */
		uns i;
		for (i = 0; ints[i].first_char != I->last_char; i++);
		ints[i].first_char = I->first_char;
		ints[i].context_len += I->context_len;
		ints[i].start_type = I->start_type;
		TRACE("Merged with interval %d on the right side", i);
		if (I != &tmp_int)
		{
			ASSERT(I == ints + (i-1));
			contint_delete(ints, maxints, i-1);
		}
		I = ints+i;
	}
	if (I == &tmp_int)
	{
		/* New interval.  */
		uns i, right_neighbour = 0;
		for (i = 0; i < maxints && ints[i].first_char; i++)
			if (ints[i].first_char < tmp_int.first_char)
				right_neighbour = i + 1;
		if (i >= maxints)
		{
			TRACE("Cannot add a new interval, since the maximum %d has already been reached", maxints);
			return 0;
		}
		contint_insert(ints, maxints, right_neighbour);
		ints[right_neighbour] = tmp_int;
		TRACE("Added a new interval before %d", right_neighbour);
	}
	return total_added;
}

static void
dump_context(struct query *q, struct result_note *note, enum text_mode mode, uns type, uns max_intervals, int context)
{
	/* Select only positions pointing to our position-space.  */
	uns noted_pos[HARD_MAX_NOTES+1];
	uns count = 0;
	TRACE("dump_context() entered in mode %c, type %d (%s), intervals %d, context %d",
		tm_letter[mode], type,
		type==UNSET_TYPE
			? "unset"
			: (char*) (mode==TM_META ? mt_names[type] : wt_names[type]),
		max_intervals, context);
	for (uns i=0; i<HARD_MAX_NOTES && note->best[i] != 0xffff; i++)
	{
		noted_pos[count] = result_note_pos(note->best[i], mode, type);
		if (noted_pos[count] != RESNOTE_INVALID_POS)
		{
			TRACE("Filtered best %d: pos %d", count, noted_pos[count]);
			count++;
		}
	}
	noted_pos[count] = RESNOTE_INVALID_POS;

	/* Verbose debug messages.  */
	if (0)
	{
		TRACE("Length of text: %d", wptr.text_end-wptr.text);
		char mask[20];
		sprintf(mask, "Text: %%%d.%ds", wptr.text_end-wptr.text, wptr.text_end-wptr.text);
		TRACE(mask, wptr.text);
		for (lexmap_first(); !lexmap_eof(); lexmap_next())
		{
			char mask[20];
			sprintf(mask, "Word %%3d: l%%2d c%%x \"%%%d.%ds\"",
				wptr.curr_word->olen, wptr.curr_word->olen);
			TRACE(mask, wptr.curr_word->pos, wptr.curr_word->olen, wptr.curr_word->type, wptr.curr_word->orig);
		}
	}

#define	ADD_CONTEXT(start, end, how_much, message) { \
	uns added_context = contint_add(q, start, end, interval, max_intervals, how_much); \
	context -= added_context; \
	if (added_context > 0) \
		TRACE message; \
}
	/* Find optimal context intervals.  */
	struct context_interval interval[max_intervals];
	bzero(interval, max_intervals * sizeof(struct context_interval));
	if (count > 0)
	{
		int context1 = context / count;
		uns i;
		/* Distribute the available context approximately uniformely
		 * among the best matches.  */
		for (i=0; i<count && context > 0; i++)
		{
			lexmap_find_word(noted_pos[i]);
			if (lexmap_eof())
				continue;
			if (wptr.curr_word->pos == noted_pos[i])
				ADD_CONTEXT(wptr.curr_word->orig, wptr.curr_word->orig + wptr.curr_word->olen, context1,
					("...Added best interval of length %d", added_context));
		}
		/* If we still have an available context, try to spend it
		 * again, preferring the former intervals.  */
		for (i=0; i < max_intervals && interval[i].first_char && context > 0; i++)
			ADD_CONTEXT(interval[i].first_char, interval[i].last_char, context/2,
				("...Stretched best interval %d by length %d", i, added_context));
	}
	if (!interval[0].first_char)
	{
		lexmap_first();
		if (!lexmap_eof())
		{
			ADD_CONTEXT(wptr.curr_word->orig, wptr.curr_word->orig + wptr.curr_word->olen, context,
				("...Added beginning of text of length %d", added_context));
		}
		else if (wptr.text_end > wptr.text)	/* might consist of non-words only */
		{
			byte *pos = wptr.text;
			uns c;
			GET_TAGGED_CHAR(pos, c);	/* skip the first change of type */
			if (pos < wptr.text_end)
			{
				byte *pos1 = pos;
				GET_TAGGED_CHAR(pos1, c);
				ADD_CONTEXT(pos, pos1, context,
					("...Added beginning of non-word text of length %d", added_context));
			}
		}
	}
	while (context > 0)
	{
		int save_context = context;
		for (uns i=0; i < max_intervals && interval[i].first_char && context > 0; i++)
			ADD_CONTEXT(interval[i].first_char, interval[i].last_char, context,
				("...Finally, stretched interval %d by length %d", i, added_context));
		if (context == save_context)
			break;
	}

	/* Dump the titles and the highlited context.  */
	for (uns i=0; i<max_intervals && interval[i].first_char; i++)
	{
		TRACE("Result: Interval %d of type %x, bytes %d-%d context %d", i,
			interval[i].start_type, interval[i].first_char - wptr.text,
			interval[i].last_char - wptr.text, interval[i].context_len);
		dump_text_interval(noted_pos, mode, interval + i);
	}
}

/***** Dumper of URL records *****/

struct save_lm_pos {		/* here we remember the word-numbering of the lex-mapper */
	word meta[MT_MAX];
	word word;
};

struct card_url {
	uns id;			/* just for debugging */
	byte *start, *end, *suffix;
	uns first_redirect, last_redirect;
	uns flags;
	int weight, pagerank;
	struct save_lm_pos saved_pos;
};

struct card_redirect {
	byte *start, *end;
	uns flags;
	int weight;
	struct save_lm_pos saved_pos;
};

#define	ASORT_PREFIX(x)	card_url_##x
#define	ASORT_KEY_TYPE	int
#define	ASORT_ELT(i)	-array[i].weight
#define	ASORT_SWAP(i,j)	do { struct card_url tmp=array[j]; array[j]=array[i]; array[i]=tmp; } while(0)
#define	ASORT_EXTRA_ARGS	, struct card_url *array
#include "lib/arraysort.h"

#define	ASORT_PREFIX(x)	card_redirect_##x
#define	ASORT_KEY_TYPE	int
#define	ASORT_ELT(i)	-array[i].weight
#define	ASORT_SWAP(i,j)	do { struct card_redirect tmp=array[j]; array[j]=array[i]; array[i]=tmp; } while(0)
#define	ASORT_EXTRA_ARGS	, struct card_redirect *array
#include "lib/arraysort.h"

static void
dump_html_escaped_word(byte *start, byte *end)
{
	while (start < end)
	{
		byte *c = start;
		byte *seq_name = NULL;
		while (c < end && !(seq_name = html_escape(*c) ))
			c++;
		if (c > start)
			reply_string(current_query, start, c-start);
		if (seq_name)
		{
			reply_string(current_query, seq_name, strlen(seq_name));
			c++;
		}
		start = c;
	}
}

static void
dump_highlighted_url(byte *start, byte *end, byte *tag_name)
{
	/* Deescape the URL to not confuse e.g. ~ == %7E with words.  */
	byte original_url[end-start+1], deescaped_url[end-start+1];
	memcpy(original_url, start, end-start);
	original_url[end-start] = 0;
	url_deescape(original_url, deescaped_url);
	start = deescaped_url;
	end = start + strlen(start);

	lexmap_parse(start, end, 0, 1);
	/* The input is just an ordinary ASCII-text without control sequences.  */
	byte buf[20];
	sprintf(buf, "M<%s>", tag_name);
	reply_string(current_query, buf, strlen(buf));
	byte *dumped = start;
	for (lexmap_first(); !lexmap_eof(); lexmap_next())
	{
		struct lexmap_word *w = wptr.curr_word;
		uns found;
		dump_html_escaped_word(dumped, w->orig);
		if ((found = hilhash_found()))
			reply_string(current_query, "<found>", 7);
		reply_string(current_query, w->orig, w->olen);
		if (found)
			reply_string(current_query, "</found>", 8);
		dumped = w->orig + w->olen;
	}
	dump_html_escaped_word(dumped, end);
	reply_f("</%s>", tag_name);
}

static void
dump_lines_between(struct query *q, struct result_note *note, struct save_lm_pos *saved_pos, byte *start, byte *end)
{
	struct parsed_attr pa;
	for (byte *record = start; get_attr(&record, end, &pa) > 0; )
	{
		if (pa.attr == 'M')
		{
			uns weight = 0;
			byte *a = pa.val;
			if (*a >= '0' && *a <= '3')
				weight = *a++ - '0';
			ASSERT(*a >= 0x90 && *a < 0xa0);
			uns type = *a & 0x0f;
			uns know_pos = saved_pos && saved_pos->meta[type] < 0xffff;
			lexmap_parse(a, pa.val + pa.len, know_pos ? saved_pos->meta[type] : 0, 0);
			if (know_pos)
				saved_pos->meta[type] = MIN(lm_pos, 0xffff);
			dump_context(q, note, TM_META, know_pos ? type : UNSET_TYPE, 2, q->title_chars);
		}
		else if (pa.attr == 'X')
		{
			uns know_pos = saved_pos && saved_pos->word < 0xffff;
			lexmap_parse(pa.val, pa.val + pa.len, know_pos ? saved_pos->word : 0, 0);
			if (know_pos)
				saved_pos->word = MIN(lm_pos, 0xffff);
			dump_context(q, note, TM_WORD, UNSET_TYPE, q->intervals, q->context_chars);
		}
		else
		{
			byte tmp = pa.attr;
			reply_string(current_query, &tmp, 1);
			reply_string(current_query, pa.val, pa.len);
			reply_string(current_query, "\n", 1);
		}

		if (pa.attr == 'U')
			dump_highlighted_url(pa.val, pa.val + pa.len, "url");
		else if (pa.attr == 'y')
			dump_highlighted_url(pa.val, pa.val + pa.len, "redirect");
		else if (pa.attr == 'b')
			dump_highlighted_url(pa.val, pa.val + pa.len, "frameof");
	}
}

static void
dump_card_url(struct query *q, struct result_note *note, struct card_url *url, struct card_redirect *reds)
{
	uns r_cnt = url->last_redirect - url->first_redirect;
	if (r_cnt <= 1)		/* Does not cut the redirect if global_redirect_url_max==0, but who cares.  */
	{
		dump_lines_between(q, note, &url->saved_pos, url->start, url->end);
		return;
	}
	dump_lines_between(q, note, &url->saved_pos, url->start, reds[ url->first_redirect ].start);
	card_redirect_sort(r_cnt, reds + url->first_redirect);
	for (uns i=0; i<r_cnt && i<global_redirect_url_max; i++)
	{
		struct card_redirect *r = reds + url->first_redirect + i;
		dump_lines_between(q, note, &r->saved_pos, r->start, r->end);
	}
	ASSERT(url->suffix <= url->end-2);
	dump_lines_between(q, note, NULL, url->suffix, url->end);
}

/***** The body of the card-dumper *****/

/* Bonuses for URL's weight: */
#define	FLAG_REDIR		1
#define	FLAG_REDIR_FOUND	2
#define	FLAG_CAT		4
#define	FLAG_CAT2		8
#define	FLAG_BEST_META		16
#define	FLAG_FOUND_META		32

#define	WEIGHT_REDIR		200
#define	WEIGHT_REDIR_FOUND	500
#define	WEIGHT_CAT		8000
#define	WEIGHT_CAT2		4000
#define	WEIGHT_BEST_META	9000
#define	WEIGHT_FOUND_META	2000

#define	WEIGHT_FOUND_MATCH	5000
#define	WEIGHT_STRLEN		-1
#define	WEIGHT_ORDER		-20
#define	WEIGHT_PAGERANK		0

static void
show_card(struct query *q, struct result_note *note, byte *card_start, byte *card_end, struct hilite_hash *hilhash)
{
	struct card_attr *ca = note->attr;
	oid_t oid;
	struct database *db = attr_to_db(ca, &oid);
	uns url_count, red_count, x_count;
	struct card_url *urls;
	struct card_redirect *reds, *xs;

	db_switch_config(db);
	show_card_header(q, db, oid, note);

	/* Count and allocate the arrays for URL's and their redirects.  */
	uns nested = 0;
	url_count = red_count = x_count = 0;
	byte *text_starts = card_end, *url_starts = NULL, *x_starts = NULL;
	struct parsed_attr pa;
	byte *start_attr;
	for (byte *attr = card_start; start_attr=attr, get_attr(&attr, card_end, &pa) >= 0; )
	{
		if (pa.attr == '(')
		{
			if (pa.val[0] == 'x')
			{
				text_starts = start_attr;
				break;
			}
			nested++;
			if (!url_starts)
				url_starts = start_attr;
		}
		else if (!nested)
		{
			text_starts = start_attr;
			break;
		}
		else if (!pa.attr)
			ASSERT(0);
		else if (pa.attr == ')')
		{
			if (nested == 1)
				url_count++;
			else if (nested == 2)
				red_count++;
			else
				ASSERT(0);
			nested--;
		}
	}
	ASSERT(!nested);
	if (!url_starts)
		url_starts = text_starts;
	for (byte *attr = text_starts; start_attr=attr, get_attr(&attr, card_end, &pa) >= 0; )
	{
		if (pa.attr == '(')
		{
			ASSERT(pa.val[0] == 'x');
			nested++;
			if (!x_starts)
				x_starts = start_attr;
		}
		else if (pa.attr == ')')
		{
			ASSERT(nested == 1);
			x_count++;
			nested--;
		}
	}
	if (!x_starts)
		x_starts = card_end;
	ASSERT(!nested);
	urls = url_count > 0 ? xmalloc(url_count * sizeof(struct card_url)) : NULL;
	reds = red_count > 0 ? xmalloc(red_count * sizeof(struct card_redirect)) : NULL;
	xs = x_count > 0 ? xmalloc(x_count * sizeof(struct card_redirect)) : NULL;

	TRACE("OID %08x: %d urls, %d redirects, and %d reftexts", oid, url_count, red_count, x_count);
	for (uns i=0; i<HARD_MAX_NOTES && note->best[i] != 0xffff; i++)
		TRACE("Best %d: %04x", i, note->best[i]);

	/* Dump the beginning of the card and shift saved_pos accordingly.  */
	struct save_lm_pos saved_pos;
	bzero(&saved_pos, sizeof(struct save_lm_pos));
	wptr.hh = hilhash;	/* initialize the lex-mapper */
	wptr.db = db;
	dump_lines_between(q, note, &saved_pos, card_start, url_starts);

	/* Fill urls and reds:
	 *   - for every URL and redirect, the start and end is recorded and
	 *     the weight is computed
	 *   - for every URL, the pointer to the first and last redirect is
	 *     stored, and saved_pos is saved.
	 * Meanwhile, shift saved_pos accordingly.  */
	uns url_idx = 0, red_idx = 0;
	for (byte *attr = card_start; start_attr=attr, get_attr(&attr, text_starts, &pa) >= 0; )
	{
		uns found_words;
		switch (pa.attr)
		{
			case '(':
				nested++;
				if (nested == 1)
				{
					ASSERT(pa.val[0] == 'U' && pa.len == 1);
					urls[url_idx].id = url_idx;
					urls[url_idx].start = start_attr;
					urls[url_idx].first_redirect = red_idx;
					urls[url_idx].flags = 0;
					urls[url_idx].weight = urls[url_idx].pagerank = 0;
					urls[url_idx].saved_pos = saved_pos;
				}
				else if (nested == 2)
				{
					ASSERT(pa.val[0] == 'y' && pa.len == 1);
					reds[red_idx].start = start_attr;
					reds[red_idx].flags = 0;
					reds[red_idx].weight = 0;
					reds[red_idx].saved_pos = saved_pos;
				}
				else
					ASSERT(0);
				break;
			case 'y':
			case 'U':
				lexmap_parse(pa.val, pa.val + pa.len, 0, 1);
				found_words = hilhash_count_found_words();
				int tmp_weight = WEIGHT_STRLEN * pa.len
					+ WEIGHT_FOUND_MATCH * !!found_words;
				if (nested == 1)
				{
					ASSERT(pa.attr == 'U');
					urls[url_idx].weight += tmp_weight + WEIGHT_ORDER * url_idx;
					TRACE("URL %d: strlen %d, found %d", url_idx, pa.len, found_words);
				}
				else if (nested == 2)
				{
					ASSERT(pa.attr == 'y');
					reds[red_idx].weight += tmp_weight + WEIGHT_ORDER * (red_idx - urls[url_idx].first_redirect);
					TRACE("Redirect %d: order %d, strlen %d, found %d", red_idx, red_idx - urls[url_idx].first_redirect + 1, pa.len, found_words);
					urls[url_idx].flags |= FLAG_REDIR | (found_words ? FLAG_REDIR_FOUND : 0);
				}
				else
					ASSERT(0);
				break;
			case 'K':
				if (nested == 2)
				{
					if (reds[red_idx].flags & FLAG_CAT)
						reds[red_idx].flags |= FLAG_CAT2;
					else
						reds[red_idx].flags |= FLAG_CAT;
					TRACE("Redirect %d: found catalog category", red_idx);
				}
				if (urls[url_idx].flags & FLAG_CAT)
					urls[url_idx].flags |= FLAG_CAT2;
				else
					urls[url_idx].flags |= FLAG_CAT;
				TRACE("URL %d: found catalog category", url_idx);
				break;
			case 'W':
				tmp_weight = atoi(pa.val+1);		// skip one-letter type
				if (nested == 2)
				{
					reds[red_idx].weight += WEIGHT_PAGERANK * tmp_weight;
					TRACE("Redirect %d: found pagerank %d", red_idx, tmp_weight);
				}
				urls[url_idx].pagerank = MAX(urls[url_idx].pagerank, tmp_weight);
				TRACE("URL %d: pagerank updated by %d", url_idx, tmp_weight);
				break;
			case 'M':
				/* Meta information corresponding to the URL-record.
				 * It contains NO type change.
				 *
				 * Some metas in the card might have not been indexed by chewer (see
				 * Chewer.GiantBanMeta), however we will not disturb anything by
				 * indexing them now, since each meta-type is either completely
				 * indexed or completely unindexed.
				 * */
				; byte *a = pa.val;
				uns flags = 0;
				tmp_weight = 0;
				if (*a >= '0' && *a <= '3')
					tmp_weight = *a++ - '0';
				ASSERT(*a >= 0x90 && *a < 0xa0);
				uns type = *a & 0x0f;
				uns know_pos = saved_pos.meta[type] < 0xffff;
				lexmap_parse(a, pa.val + pa.len, saved_pos.meta[type], 0);
				found_words = hilhash_count_found_words();
				if (found_words)
					flags |= FLAG_FOUND_META;
				if (nested == 1)
					TRACE("URL %d: meta %d of weight %d has %d found words", url_idx, type, tmp_weight, found_words);
				else
					TRACE("Redirect %d: meta %d of weight %d has %d found words", red_idx, type, tmp_weight, found_words);
				if (know_pos && lm_pos > saved_pos.meta[type])
				{
					uns best_matches = 0;
					for (uns i=0; i<HARD_MAX_NOTES && note->best[i] != 0xffff; i++)
					{
						uns pos = result_note_pos(note->best[i], TM_META, type);
						if (pos != RESNOTE_INVALID_POS
						&& pos >= saved_pos.meta[type] && pos < lm_pos)
							best_matches++;
					}
					if (best_matches)
						flags |= FLAG_BEST_META;
					if (nested == 1)
						TRACE("URL %d: meta %d of weight %d has %d best matches", url_idx, type, tmp_weight, best_matches);
					else
						TRACE("Redirect %d: meta %d of weight %d has %d best matches", red_idx, type, tmp_weight, best_matches);
					saved_pos.meta[type] = MIN(lm_pos, 0xffff);
				}
				if (nested == 2)
					reds[red_idx].flags |= flags;
				urls[url_idx].flags |= flags;
				break;
			case ')':
				ASSERT(pa.len == 0);
				int *Weight;
				struct card_url *u;
				struct card_redirect *r;
				if (nested == 2)
				{
					u = NULL;
					r = reds + red_idx;
					r->end = attr;
					flags = r->flags;
					Weight = &r->weight;
					red_idx++;
				}
				else if (nested == 1)
				{
					r = NULL;
					u = urls + url_idx;
					if (red_idx > urls[url_idx].first_redirect)
						u->suffix = reds[red_idx-1].end;
					else
						u->suffix = start_attr;
					u->end = attr;
					u->last_redirect = red_idx;
					flags = u->flags;
					Weight = &u->weight;
					*Weight += WEIGHT_PAGERANK * u->pagerank;
					url_idx++;
				}
				else
					ASSERT(0);

				if (flags & FLAG_REDIR_FOUND)
					*Weight += WEIGHT_REDIR_FOUND;
				else if (flags & FLAG_REDIR)
					*Weight += WEIGHT_REDIR;
				if (flags & FLAG_CAT2)
					*Weight += WEIGHT_CAT2;
				else if (flags & FLAG_CAT)
					*Weight += WEIGHT_CAT;
				if (flags & FLAG_BEST_META)
					*Weight += WEIGHT_BEST_META;
				else if (flags & FLAG_FOUND_META)
					*Weight += WEIGHT_FOUND_META;

				if (nested == 2)
					TRACE("Redirect %d finished: order %d, start %p, length %d, weight %d",
						red_idx-1, red_idx - urls[url_idx].first_redirect, r->start, r->end - r->start, r->weight);
				else
					TRACE("URL %d finished: id %d, start %p, length %d, pagerank %d, weight %d, redirects %d-%d",
						url_idx-1, u->id, u->start, u->end - u->start,
						u->pagerank, u->weight, u->first_redirect, u->last_redirect);
				nested--;
				break;
		}
	}
	ASSERT(url_idx == url_count && red_idx == red_count);
	/* Fill xs */
	uns x_idx = 0;
	for (byte *attr = x_starts; start_attr=attr, get_attr(&attr, card_end, &pa) >= 0; )
	{
		uns found_words;
		switch (pa.attr)
		{
			case '(':
				ASSERT(pa.val[0] == 'x' && pa.len == 1);
				xs[x_idx].start = start_attr;
				xs[x_idx].flags = 0;
				xs[x_idx].weight = 0;
				xs[x_idx].saved_pos = saved_pos;
				break;
			case 'W':
				; int tmp_weight = atoi(pa.val+1);		// skip one-letter type
				xs[x_idx].weight += WEIGHT_PAGERANK * tmp_weight;
				TRACE("Reftext %d: found rank %d", x_idx, tmp_weight);
				break;
			case 'M':
				; byte *a = pa.val;
				uns flags = 0;
				tmp_weight = 0;
				if (*a >= '0' && *a <= '3')
					tmp_weight = *a++ - '0';
				ASSERT(*a >= 0x90 && *a < 0xa0);
				uns type = *a & 0x0f;
				uns know_pos = saved_pos.meta[type] < 0xffff;
				lexmap_parse(a, pa.val + pa.len, saved_pos.meta[type], 0);
				found_words = hilhash_count_found_words();
				if (found_words)
					flags |= FLAG_FOUND_META;
				TRACE("Reftext %d: meta %d of weight %d has %d found words", x_idx, type, tmp_weight, found_words);
				if (know_pos && lm_pos > saved_pos.meta[type])
				{
					uns best_matches = 0;
					for (uns i=0; i<HARD_MAX_NOTES && note->best[i] != 0xffff; i++)
					{
						uns pos = result_note_pos(note->best[i], TM_META, type);
						if (pos != RESNOTE_INVALID_POS
						&& pos >= saved_pos.meta[type] && pos < lm_pos)
							best_matches++;
					}
					if (best_matches)
						flags |= FLAG_BEST_META;
					TRACE("Reftext %d: meta %d of weight %d has %d best matches", x_idx, type, tmp_weight, best_matches);
					saved_pos.meta[type] = MIN(lm_pos, 0xffff);
				}
				xs[x_idx].flags |= flags;
				break;
			case ')':
				ASSERT(pa.len == 0);
				struct card_redirect *r = xs + x_idx;
				r->end = attr;
				if (r->flags & (FLAG_BEST_META | FLAG_FOUND_META))
					r->weight += WEIGHT_BEST_META;
				TRACE("Reftext %d finished: start %p, length %d, weight %d",
					x_idx, r->start, r->end - r->start, r->weight);
				x_idx++;
				break;
		}
	}
	ASSERT(x_idx == x_count);

	/* Dump URL's.  */
	if (url_count > 0)
	{
		ASSERT(urls[0].start == url_starts && urls[url_count-1].end == text_starts);
		if (url_count > 1)
			card_url_sort(url_count, urls);
		for (uns i=0; i<url_count && i<q->url_max; i++)
		{
			TRACE("Dumping URL#%d with id %d and weight %d", i, urls[i].id, urls[i].weight);
			dump_card_url(q, note, urls + i, reds);
		}
	}

	/* Dump the rest of the card, i.e. the meta-information corresponding
	 * to the document and its body.  */
	dump_lines_between(q, note, &saved_pos, text_starts, x_starts);

	/* Dump reftexts.  */
	if (x_count > 0)
	{
		if (x_count > 1)
			card_redirect_sort(x_count, xs);
		for (uns i=0; i<x_count && (xs[i].weight >= WEIGHT_BEST_META || q->context_chars == CONTEXT_FULL); i++)
		{
			struct card_redirect *x = xs + i;
			TRACE("Dumping Reftext#%d with weight %d", i, x->weight);
			dump_lines_between(q, note, &x->saved_pos, x->start, x->end);
		}
	}

	TRACE("Total words in card: %d", saved_pos.word);
	for (uns i=0; i<MT_MAX; i++)
		if (saved_pos.meta[i] > 0)
			TRACE("Total metas of type %d (%s) in card: %d", i, mt_names[i], saved_pos.meta[i]);
	if (urls)
		xfree(urls);
	if (reds)
		xfree(reds);
	if (xs)
		xfree(xs);

	reply_f("%s", "");			/* Avoid format string warning */
}

static void
show_results_full(struct query *q)
{
  struct results *res = q->results;
  uns maxres = MIN(res->nresults, max_output_matches);
  struct val_set *rng;
  struct result_note *notes[maxres];
  struct mmap_request *mmaps[maxres];
  struct hilite_hash *hilhash;
  uns nres = 0;
  uns i;

  profiler_switch(&prof_resf);
  for (rng=q->range; rng; rng=rng->next)
    {
      uns from = rng->min;
      uns to = MIN(rng->max, res->nresults);
      while (from <= to && nres < maxres)
	notes[nres++] = res->result_heap[from++];
    }
  struct mmap_request mmap_array[nres];
  for (i=0; i<nres; i++)
    {
      struct mmap_request *m = &mmap_array[i];
      struct card_attr *ca = notes[i]->attr;
      struct database *db = attr_to_db(ca, NULL);
      m->u.req.fd = db->fd_cards;
      m->u.req.start = (sh_off_t) ca->card << CARD_POS_SHIFT;
      m->u.req.end = (sh_off_t) (ca+1)->card << CARD_POS_SHIFT;
      m->userdata = i;
    }
  if (mmap_regions(q, mmap_array, nres) < 0)
    {
      log(L_ERROR, "Error mapping index cards for query result display");
      return;
    }
  profiler_switch(&prof_results);
  for (i=0; i<nres; i++)
    mmaps[mmap_array[i].userdata] = &mmap_array[i];
  hilhash = hilhash_create(q, HILITE_HASH_SIZE, q->results->first_hilite);
  for (i=0; i<nres; i++)
  {
    byte *ptr;
    uns type;
    int len = lizard_memread(liz_buf, mmaps[i]->u.map.start, &ptr, &type);
    if (len < 0)
    {
      struct card_attr *ca = notes[i]->attr;
      oid_t oid;
      attr_to_db(ca, &oid);
      die("Cannot decompress object %08x: %m", oid);
    }
    get_attr_set_type(type);
    show_card(q, notes[i], ptr, ptr+len, hilhash);
  }
}

void
show_results(struct query *q)
{
  if (!q->results->nresults || q->list_only > 1)
    ;	/* No results or stats only */
  else if (q->list_only)
    show_results_list(q);
  else
    show_results_full(q);
}

void
cards_init(void)
{
  lm_init();
}
