/*
 *	A Simple Utility for Fast Transfer of Large Files: Received
 *
 *	(c) 2002 Martin Mares <mj@ucw.cz>
 *
 *	For the protocol, see comments in file-recv.c.
 */

#include "sherlock/sherlock.h"
#include "lib/conf.h"
#include "lib/lfs.h"
#include "lib/md5.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>

#include <zlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

static uns timeout = 0;

static struct cfitem filerecv_config[] = {
  { "FileRecv",		CT_SECTION,	NULL },
  { "Timeout",		CT_INT,		&timeout },
  { NULL,		CT_STOP,	NULL }
};

/* Smallish chunks are more friendly to caches */
#define CHUNK_SIZE 65536

static void
alarm_handler(int UNUSED whatsit)
{
  die("Timed out.");
}

static void
sread(int sk, void *buf, int len, byte *msg)
{
  int l;
  byte *b = buf;

  while (len > 0)
    {
      alarm(timeout);
      l = read(sk, b, len);
      if (l < 0)
	die("Error receiving %s: %m", msg);
      if (!l)
	die("Error receiving %s: EOF", msg);
      b += l;
      len -= l;
    }
}

int main(int argc, char **argv)
{
  int i, sk, newsk, fd, result;
  socklen_t len;
  struct sockaddr_in sa;
  socklen_t salen = sizeof(sa);
  byte hostname[256], keybin[MD5_SIZE], key[MD5_HEX_SIZE], *rxbuf;
  struct hostent *he;
  u64 size, recv_size;
  u32 comp;
  u32 adler;
  struct MD5Context md5c;
  byte md5[16], sig[16];

  log_init("file-recv");
  cf_register(filerecv_config);
  if (cf_getopt(argc, argv, CF_SHORT_OPTS, CF_NO_LONG_OPTS, NULL) >= 0 ||
      optind + 1 >= argc)
    {
      fputs("Usage: file-recv [<-C,-S as usually>] <destdir> <file> ...\n", stderr);
      return 1;
    }

  if (gethostname(hostname, sizeof(hostname)) < 0)
    die("gethostname: %m");
  if (!(he = gethostbyname(hostname)))
    die("Sir, I have severe problems with my identity. I asked to be called %s and everything I got was: %m", hostname);
  signal(SIGALRM, alarm_handler);
  if (chdir(argv[optind]) < 0)
    die("chdir(%s): %m", argv[optind]);

  for (i=optind+1; i<argc; i++)
    {
      alarm(timeout);
      if ((sk = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
	die("socket: %m");
      sa.sin_family = AF_INET;
      memcpy(&sa.sin_addr, he->h_addr, 4);
      sa.sin_port = htons(0);
      if (bind(sk, (struct sockaddr *) &sa, sizeof(sa)) < 0)
	die("bind: %m");
      if (getsockname(sk, (struct sockaddr *) &sa, &salen) < 0)
	die("getsockname: %m");
      if (listen(sk, 1) < 0)
	die("listen: %m");

      if ((fd = sh_open(argv[i], O_WRONLY|O_CREAT|O_TRUNC, 0666)) < 0)
	die("Unable to create %s: %m", argv[i]);

      randomkey(keybin, MD5_SIZE);
      md5_to_hex(keybin, key);
      printf("#FS# %s %d %s %s\n", inet_ntoa(sa.sin_addr), ntohs(sa.sin_port), key, argv[i]);
      fflush(stdout);

      len = sizeof(sa);
      if ((newsk = accept(sk, (struct sockaddr *) &sa, &len)) < 0)
	die("accept: %m");
      close(sk);
      sread(newsk, &size, 8, "file header");
      sread(newsk, &comp, 4, "file header");
      rxbuf = xmalloc(CHUNK_SIZE);
      recv_size = 0;
      if (comp)
	{
	  z_stream zs;
	  byte *outbuf = xmalloc(CHUNK_SIZE);
	  bzero(&zs, sizeof(zs));
	  if (inflateInit(&zs) != Z_OK)
	    die("inflateInit failed: %s", zs.msg);
	  /* zlib calculates its own adler32, but forgets to report it, so we need to do it ourselves :( */
	  adler = adler32(0, NULL, 0);
	  do
	    {
	      if (!zs.avail_in)
		{
		  u32 l;
		  sread(newsk, &l, sizeof(l), "file data");
		  sread(newsk, rxbuf, l, "file data");
		  zs.next_in = rxbuf;
		  zs.avail_in = l;
		}
	      if (!zs.avail_out)
		{
		  zs.next_out = outbuf;
		  zs.avail_out = CHUNK_SIZE;
		}
	      result = inflate(&zs, Z_NO_FLUSH);
	      if (result != Z_STREAM_END && result != Z_OK)
		die("inflate error: %d (%s)", result, zs.msg);
	      if ((result == Z_STREAM_END || !zs.avail_out) && zs.next_out > outbuf)
		{
		  int l = zs.next_out - outbuf;
		  int w = write(fd, outbuf, l);
		  if (w < 0)
		    die("write: %m");
		  if (w != l)
		    die("Short write: %d of %d bytes written", w, l);
		  adler = adler32(adler, outbuf, l);
		  recv_size += l;
		}
	    }
	  while (result != Z_STREAM_END);
	  inflateEnd(&zs);
	}
      else
	{
	  adler = adler32(0, NULL, 0);
	  while (size > 0)
	    {
	      uns chunk = CHUNK_SIZE;
	      if (chunk > size)
		chunk = size;
	      sread(newsk, rxbuf, chunk, "file data");
	      if (write(fd, rxbuf, chunk) != (int) chunk)
		die("write: %m");
	      adler = adler32(adler, rxbuf, chunk);
	      size -= chunk;
	      recv_size += chunk;
	    }
	}

      MD5Init(&md5c);
      MD5Update(&md5c, key, strlen(key));
      MD5Update(&md5c, (byte *) &adler, sizeof(adler));
      MD5Final(md5, &md5c);
      sread(newsk, sig, sizeof(sig), "signature");
      if (memcmp(md5, sig, sizeof(md5)))
	die("File checksum mismatch");

      log(L_INFO, "Received file %s (%Ld bytes)", argv[i], (long long) recv_size);

      close(fd);
      close(newsk);
    }

  return 0;
}
