#!/usr/bin/perl
#
#	Sherlock Query Sender
#	(c) 2001--2003 Martin Mares <mj@ucw.cz>
#

use strict;
use warnings;
use Getopt::Long;

use lib 'lib/perl5';
use Sherlock::Query();

sub do_query($);

my $host = "localhost";
my $port = 8192;
my $requests = 1;
my $headers = 1;
my $body = 1;
my $footer = 1;
my $stats = 0;
my $control = 0;
my $debug = 0;

Getopt::Long::Configure("bundling");
GetOptions(
	"host|h=s" => \$host,
	"port|p=i" => \$port,
	"silent|s" => sub { $headers=$body=$footer=$stats=$requests=0; },
	"requests!" => \$requests,
	"header!" => \$headers,
	"body!" => \$body,
	"footer!" => \$footer,
	"stats!" => \$stats,
	"control!" => \$control,
	"debug!" => \$debug,
	# Need to repeat that since getopt doesn't support "long|short!" spec
	"R" => \$requests,
	"H" => \$headers,
	"B" => \$body,
	"F" => \$footer,
	"S" => \$stats,
	"C" => \$control,
	"d" => \$debug
) || die "Syntax: query [<options>] [<query>]

Options:
-h, --host=hostname	Select search server host
-p, --port=port		Select search server port
-s, --silent		--norequests --noheaders --nobody --nofooter --nostats
-R, --requests		Show requests (default: on)
-H, --header		Show reply headers (default: on)
-B, --body		Show reply body (default: on)
-F, --footer		Show reply footer (default: on)
-S, --stats		Show statistics
-C, --control		Send a control command instead of query
-d, --debug		Show a debugging dump
";

my $total_ok = 0;
my $total_err = 0;
my $total_time = 0;
my $total_cached = 0;
my %totals = ();
if (@ARGV) {
	my $query = join(' ', @ARGV);
	do_query($query);
} else {
	while (<>) {
		chomp;
		do_query($_);
	}
}
if ($stats) {
	print "## Total queries: $total_ok OK + $total_err FAILED\n";
	print "## Total time: $total_time\n";
	print "## Cache hits: $total_cached\n";
	foreach my $k (sort keys %totals) {
		print "## Total prof_$k: ", $totals{$k}, "\n";
	}
}
exit $total_err ? 1 : 0;

sub
do_query($)
{
	my $query = shift @_;
	my $q = Sherlock::Query->new("$host:$port");
	my $stat;

	if ($control) {
		$query = "control $query";
		print "<<< $query\n" if $requests;
		$stat = $q->command($query);
		$q->print if $debug;
		local $, = "\n";
		print "$stat\n" if $headers;
		if ($body) {
			foreach my $c (@{$q->{RAW}}) {
				print @$c, "\n";
			}
		}
	} else {
		print "<<< $query\n" if $requests;
		$stat = $q->query($query);
		$q->print if $debug;
		local $, = "\n";
		print "$stat\n" if $headers;
		print @{$q->{HEADER}{RAW}}, "\n" if $headers;
		if ($body) {
			foreach my $c (@{$q->{CARDS}}) {
				print @{$c->{RAW}}, "\n";
			}
		}
		print @{$q->{FOOTER}{RAW}}, "\n" if $footer;
	}
	$total_ok++ if $stat =~ /^\+/;
	$total_err++ if $stat =~ /^-/;
	$total_cached++ if $q->{HEADER}{'C'};
	if (my $rt = $q->{FOOTER}{'t'}) { $total_time += $rt; }
	if (my $rT = $q->{FOOTER}{'T'}) { $rT =~ s/(\w+)=([0-9.]+)/$totals{$1}+=$2/ge; }
}
