#!/usr/bin/perl
# Calculate and plot timing statistics from search server logs
# (c) 2002 Martin Mares <mj@ucw.cz>

use strict;
use warnings;

my $prefix = $ARGV[0] ? $ARGV[0] : "sherlockd";
my $graph = "log/graph/$prefix";
my $relgraph = $prefix;
my $temp = "tmp/stat-$prefix";

my $lastq = undef;
my $slots = 1440/5;
my $slot_len = 86400/$slots;
my @num_queries;
my @total_time;
my @max_time;
foreach my $t (0 .. $slots-1) {
	$num_queries[$t] = $total_time[$t] = $max_time[$t] = 0;
}

while (<STDIN>) {
	chomp;
	/^I ....-..-.. (..):(..):.. (.*)/ || next;
	my $slot = int((3600*$1+60*$2)/$slot_len);
	$_ = $3;
	if (/ < (.*)/) {
		$lastq = $1;
	} elsif (my ($t) = /^> \d+ t=(\d+)/) {
		if ($lastq) {
			$num_queries[$slot]++;
			$total_time[$slot] += $t;
			$max_time[$slot] = $t if $t > $max_time[$slot];
		}
	}
}

sub add_graph($$$);

my $total_queries = 0;
my $rawfile = "$temp.raw";
open(T, ">$rawfile") or die "Unable to write $rawfile";
foreach my $t (0 .. $slots-1) {
	$total_queries += $num_queries[$t];
	my $qps = $num_queries[$t] / $slot_len;
	my $iqi = "?";
	my $avgtime = "?";
	my $maxtime = "?";
	if ($num_queries[$t]) {
		$iqi = $slot_len / $num_queries[$t];
		$avgtime = $total_time[$t] / $num_queries[$t];
		$maxtime = $max_time[$t];
	}
	my $load = $total_time[$t] / (1000 * $slot_len);
	my $hour = $t / ($slots/24);
	print T "$hour	$load	$qps	$iqi	$avgtime	$maxtime\n";
}
close T;

open(HTML, ">$graph.html") or die "Unable to write $graph.html";
my $today = `date`;
chomp $today;
print HTML <<EOF
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html40/strict.dtd">
<HTML><HEAD><TITLE>Search server statistics for $prefix</TITLE></HEAD>
<BODY>
<H1>Statistics for $prefix</H1>
<P>Calculated on $today
<P>Total queries: $total_queries
<P>Data averaged over $slot_len second slots
EOF
;
add_graph("load", "Average load", "plot \"$rawfile\" using 1:2 title \"load\"");
add_graph("qps", "Queries per second", "plot \"$rawfile\" using 1:3 title \"qps\"");
add_graph("iqi", "Average inter-query interval", "plot \"$rawfile\" using 1:4 title \"iqi [s]\"");
add_graph("avgtime", "Average query processing time", "plot \"$rawfile\" using 1:5 title \"avgtime [ms]\"");
add_graph("maxtime", "Maximal query processing time", "plot \"$rawfile\" using 1:6 title \"maxtime [ms]\"");
print HTML <<EOF
</BODY></HTML>
EOF
;
close HTML;
unlink $rawfile;
exit 0;

sub add_graph($$$) {
	my $graphname = shift @_;
	my $title = shift @_;
	my $plotcmd = shift @_;
	my $outf = "$graph-$graphname.png";
	my $relf = "$relgraph-$graphname.png";
	my $gplf = "$temp-$graphname.gnuplot";
	open X, ">$gplf" or die "Unable to write $gplf";
	print X "set terminal png\n";
	print X "set output \"$outf\"\n";
	print X "set pointsize 2\n";
	print X "set xtics 1\n";
	print X "set grid\n";
	print X "set data style lines\n";
	print X "$plotcmd\n";
	close X;
	`gnuplot $gplf`;
	$? and die "gnuplot failed with rc=$?";
	print HTML "<H2>$title</H2>\n<img src=\"$relf\" alt=\"$title\">\n";
	unlink $gplf;
}
