/*
 *	Sherlock Filter Engine -- Builtin Functions and Variables
 *
 *	(c) 1999--2006 Martin Mares <mj@ucw.cz>
 *	(c) 2001--2003 Robert Spalek <robert@ucw.cz>
 */

#include "sherlock/sherlock.h"
#include "lib/mempool.h"
#include "lib/url.h"
#include "filter/filter.h"

#ifdef CONFIG_CUSTOM_FILTER
#include CONFIG_CUSTOM_FILTER
#else
#define CUSTOM_FILTER_BUILTINS
#endif

#include <errno.h>
#include <stdlib.h>
#include <string.h>

/*
 *  Functions
 */

static void
fv_str2int(struct filter_args *args UNUSED, struct filter_value *ret, struct filter_value arg[MAX_FUNC_ARGS])
{
	if (arg[0].undef)
		ret->undef = 1;
	else
	{
		char *c;
		errno = 0;
		ret->v.i = strtoul(arg[0].v.s, &c, 0);
		if (c && *c || errno == ERANGE)
			ret->undef = 1;
		else
			ret->undef = 0;
	}
}

#define	MAX_REGEX_BUF	1024

static void
fv_replace(struct filter_args *args, struct filter_value *ret, struct filter_value arg[MAX_FUNC_ARGS])
{
	byte buf[MAX_REGEX_BUF];
	int res;
	ret->undef = 1;
	if (arg[0].undef || arg[1].undef || arg[2].undef)
		return;
	res = rx_subst(arg[1].v.r->regex, arg[2].v.s, arg[0].v.s, buf, MAX_REGEX_BUF);
	if (res < 0)
	{
		log(L_ERROR, "Cannot perform fv_replace, buffer of length %d is too short for %s -> %s", MAX_REGEX_BUF, arg[0].v.s, arg[2].v.s);
	}
	else if (res == 0)
	{
		ret->undef = arg[3].undef;
		ret->v.s = arg[3].v.s;
	}
	else
	{
		ret->undef = 0;
		ret->v.s = mp_alloc(args->pool, strlen(buf)+1);
		strcpy(ret->v.s, buf);
	}
}

static void
fv_has_repeated_component(struct filter_args *args UNUSED, struct filter_value *ret, struct filter_value arg[MAX_FUNC_ARGS])
{
	if (arg[0].undef)
	{
		ret->undef = 1;
		return;
	}
	ret->undef = 0;
	ret->v.i = url_has_repeated_component(arg[0].v.s);
}

static void
fv_standardize_domain(struct filter_args *args UNUSED, struct filter_value *ret, struct filter_value arg[MAX_FUNC_ARGS])
{
	if (arg[0].undef)
	{
		ret->undef = 1;
		return;
	}
	ret->undef = 0;
	byte *domain = arg[0].v.s;
	if (*domain == '.')
		domain++;
	if (!strncasecmp(domain, "www.", 4))
		domain += 4;
	ret->v.s = mp_alloc(args->pool, strlen(domain)+2);
	ret->v.s[0] = '.';
	strcpy(ret->v.s+1, domain);
}

struct filter_function filter_builtin_func[] = {
	{ "toint",	1, {F_ET_STRING},				F_ET_INT,	fv_str2int },
	{ "replace",	4, {F_ET_STRING, F_ET_REGEXP, F_ET_STRING, F_ET_STRING},
									F_ET_STRING,	fv_replace },
	{ "has_repeated_component",	1,	{F_ET_STRING},		F_ET_INT,	fv_has_repeated_component },
	{ "standardize_domain",		1,	{F_ET_STRING},		F_ET_STRING,	fv_standardize_domain },
	CUSTOM_FILTER_BUILTINS
	{ NULL,		0, {},						0,		NULL }
};

/*
 *  Variables. Remember to update filter-test.c accordingly.
 */

struct filter_variable filter_builtin_vars[] = {
  /* URL and its parts */
  { "url",		F_LVC_RAW,	F_ET_STRING,	1 },
  { "protocol",		F_LVC_RAW,	F_ET_STRING,	1 },
  { "host",		F_LVC_RAW,	F_ET_STRING,	1 },
  { "port",		F_LVC_RAW,	F_ET_INT,	1 },
  { "path",		F_LVC_RAW,	F_ET_STRING,	1 },
  { "username",		F_LVC_RAW,	F_ET_STRING,	0 },
  { "password",		F_LVC_RAW,	F_ET_STRING,	0 },
  /* Gatherer attributes */
  { "server_type",	F_LVC_RAW,	F_ET_STRING,	1 },
  { "content_type",	F_LVC_RAW,	F_ET_STRING,	0 },
  { "content_encoding",	F_LVC_RAW,	F_ET_STRING,	0 },
  { "ignore_links",	F_LVC_RAW,	F_ET_INT,	0 },
  { "ignore_text",	F_LVC_RAW,	F_ET_INT,	0 },
  { "error_code",	F_LVC_RAW,	F_ET_INT,	0 },
  { "user_mark",	F_LVC_RAW,	F_ET_INT,	0 },
  /* Sections, limits and queueing */
  { "section",		F_LVC_RAW,	F_ET_INT,	0 },
  { "section_soft_max",	F_LVC_RAW,	F_ET_INT,	0 },	/* gatherd only */
  { "section_hard_max",	F_LVC_RAW,	F_ET_INT,	0 },	/* gatherd only */
  { "queue_bonus",	F_LVC_RAW,	F_ET_INT,	0 },
  { "queue_key",	F_LVC_RAW,	F_ET_INT,	1 },
  /* Shepherd limits and timings */
  { "soft_limit",	F_LVC_RAW,	F_ET_INT,	0 },
  { "hard_limit",	F_LVC_RAW,	F_ET_INT,	0 },
  { "fresh_limit",	F_LVC_RAW,	F_ET_INT,	0 },
  { "min_delay",	F_LVC_RAW,	F_ET_INT,	0 },
  { "max_conn",		F_LVC_RAW,	F_ET_INT,	0 },
  { "monitor",		F_LVC_RAW,	F_ET_INT,	0 },
  { "select_bonus",	F_LVC_RAW,	F_ET_INT,	0 },
  /* Indexer attributes */
  { "site",		F_LVC_RAW,	F_ET_STRING,	0 },
  { "site_level",	F_LVC_RAW,	F_ET_INT,	0 },
  { "bonus",		F_LVC_RAW,	F_ET_INT,	0 },
  { "language",		F_LVC_RAW,	F_ET_STRING,	0 },
  { "card_bonus",	F_LVC_RAW,	F_ET_INT,	0 },
  { "title",		F_LVC_RAW,	F_ET_STRING,	1 },
  { "image_size",	F_LVC_RAW,	F_ET_INT,	1 },
  { "image_aspect_ratio",F_LVC_RAW,	F_ET_INT,	1 },
  { "image_colors",	F_LVC_RAW,	F_ET_INT,	1 },
  /* Area attributes */
  { "area",		F_LVC_RAW,	F_ET_INT,	0 },
  /* Extended attributes, all attributes not listed explicitly are of type
   * string by default */
  { "s",		F_LVC_ATTR,	F_ET_INT,	1 },
  { NULL,		0,		0,		0 }
};
