#!/usr/bin/perl -w

#
# Dr. Watson -- user-friendly frontend to statistics
#
# (c) 2003-2004 Tomas Valla <tom@ucw.cz>
# (c) 2005-2006 Vladimir Jelen <vladimir.jelen@netcentrum.cz>

# How to create custom pages with graphs:
#
# As ordinary CGI script you can supply arguments to drwatson.cgi.
# Possible arguments:
#  gath_stats
#  shep_stats
#  search_stats
#  idx_stats
#  mux_stats		- write graph name into apropriate argument
#  search_pref 		- search server interfile prefixes, so what server you want to use
#  gath_all
#  exp_all
#  idx_all
#  search_all
#  mux_all		- set to nonempty if you want to view all graphs
#  sp_all
#  bdate
#  btime
#  edate
#  etime 		- date and time as in user defined time
#  force 		- force recomputing of graphs
#  time_input 		- should be one of lastday,lastweek,lastmonth,user
#  hide_office		- set to nonempty if you want to hide graph choosing screen
#
# Example:
# I'd like to view statistics of cache efficiency of sherlock0 and gatherer performance
# and spectrum for last week.
#
# drwatson.cgi?hide_office=true&time_input=lastweek&gath_stats=gatherd-performance&gath_stats=gatherd-spectrum&search_pref=sherlock0/sherlockd-&search_stats=search-cache_efficiency
#
#

use lib 'lib/perl5';
use Sherlock::CGI;
use Sherlock::Ulimit;
use Sherlock::Watsonlib;
use POSIX;
use strict;

read_config("SEARCH_PREF_MASK", "USE_SEARCH", "USE_GATHERD", "USE_SHEPHERD", "USE_INDEXER", "USE_LEX", "USE_MUX");

# if you wish to reconfigure the following stuff do it in cf/drwatson
my $use_gath = read_config_value("USE_GATHERD", 1);
my $use_shep = read_config_value("USE_SHEPHERD", 1);
my $use_search = read_config_value("USE_SEARCH", 1);
my $use_idx = read_config_value("USE_INDEXER", 1);
my $use_mux = read_config_value("USE_MUX", 1);
my $use_lex = read_config_value("USE_LEX", 1);

$use_shep = 0;
$use_mux = 0;

my $graph_cgi_url = 'graph.cgi';

my $gth_stat_mask = "graph/gatherd-*";
my $idx_stat_mask = "graph/indexer-*";
my $shep_stat_mask = "graph/shepherd-*";
my $mux_stat_mask = "graph/mux-*";
my $srch_stat_mask = "graph/search-*";

my $gatherd_prefix = "gather-";
my $indexer_prefix = "indexer-";
my $shepherd_prefix = "shepherd-";
my $mux_prefix = "mux-";
my $srch_pref_mask = read_config_value("SEARCH_PREF_MASK", "log/inter/sherlock[0-9]*");

# Log file prefix & name of stat script
my $indexer_status_prefix = "status-indexer-";
my $shepherd_status_prefix = "status-shepherd-";
my $idx_stat_status = "status-indexer";
my $shep_stat_status = "status-shepherd";

my @gath_allstats = map { (reverse split "/",$_)[0] } glob($gth_stat_mask);
my @shep_allstats= map { (reverse split "/",$_)[0] } glob($shep_stat_mask);
my @search_allstats = map { (reverse split "/",$_)[0] } glob($srch_stat_mask);
my @search_prefixes = map { (reverse split "/",$_)[0]."/sherlockd-" } glob($srch_pref_mask);
my @indexer_allstats = map { (reverse split "/",$_)[0] } glob($idx_stat_mask);
my @mux_allstats = map { (reverse split "/",$_)[0] } glob($mux_stat_mask);

push @shep_allstats, $shep_stat_status;
push @indexer_allstats, $idx_stat_status;
####

Sherlock::Ulimit::setlimit($Sherlock::Ulimit::AS, 40000000,40000000);

header("Dr. Watson");
body();
footer();

sub print_stats {
	my $stat_var = shift;
	my @r = ();

	foreach my $s (@_) {
		push @r, qq(<INPUT type="checkbox" name="$stat_var" value="$s">$s);
	}
	return @r;
}

sub header {
	my $title = shift;

	print <<END
Content-Type: text/html

<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">
<HTML>
<HEAD>
<TITLE>$title</TITLE>
</HEAD>
<BODY>
END
	;
}

sub footer {
	print "</BODY>\n</HTML>\n";
}

sub body {

	my ($bdate, $btime, $edate, $etime, $force, $time_input, $gath_all, $shep_all, $search_all, $idx_all, $mux_all, $sp_all, $hide_office);
	my (@gath_stats, @shep_stats, @search_stats, @search_pref, @idx_stats, @mux_stats);
	my %pars = (
		gath_stats => {var => \@gath_stats},
		shep_stats => {var => \@shep_stats},
		search_stats => {var => \@search_stats},
		search_pref => {var => \@search_pref},
		idx_stats => {var => \@idx_stats},
		mux_stats => {var => \@mux_stats},
		gath_all => {var => \$gath_all},
		shep_all => {var => \$shep_all},
		idx_all => {var => \$idx_all},
		search_all => {var => \$search_all},
		mux_all => {var => \$mux_all},
		sp_all => {var => \$sp_all},
		bdate => {var => \$bdate, check => '\d\d\d\d-\d{1,2}-\d{1,2}', default => '1990-01-01'},
		btime => {var => \$btime, check => '\d{1,2}:\d{1,2}', default => '00:00' },
		edate => {var => \$edate, check => '\d\d\d\d-\d{1,2}-\d{1,2}', default => strftime("%Y-%m-%d",localtime(time())) },
		etime => {var => \$etime, check => '\d{1,2}:\d{1,2}', default => '23:59' },
		force => {var => \$force },
		time_input => {var => \$time_input },
		hide_office => {var => \$hide_office },
	);
	Sherlock::CGI::parse_args(\%pars);

	if (! $hide_office) {
		print <<END
<H1>Dr. Watson's office</H1>
<PRE style="float: right">
   \\\\\\\\
   c  oo
    | .U
   __=__                        ,,,
  |.  __|___                    oo ;
  ||_/  /  /                    U= _  0
  \\_/__/__E   o                 /. .| |
   (___ ||    |~~~~~~~~~~~~~~~~'----'~|
   I---|||    |-----------------------|
   I   |||    |       c(__)           |
   ^   '--''  ^                       ^
</PRE>
<FORM method="post" action="?" enctype="application/x-www-form-urlencoded">
<H2>Give me a term</H2>
<INPUT type="radio" name="time_input" value="lastday" checked>Last day
<INPUT type="radio" name="time_input" value="lastweek">Last week
<INPUT type="radio" name="time_input" value="lastmonth">Last month
<BR>
<INPUT type="radio" name="time_input" value="user">User defined:
<BR>
<INPUT type="text" name="bdate" value="$bdate">
<INPUT type="text" name="btime" value="$btime">
Start date and time YYYY-MM-DD hh:mm<BR>

<INPUT type="text" name="edate" value="$edate">
<INPUT type="text" name="etime" value="$etime">
End date and time<BR>

<H2>What kind of examination would you like?</H2>
<TABLE border="1" cellspacing="0" cellpadding="3">
<CAPTION>Possible examinations</CAPTION>
<TR>
END
		;
		print '<TH>Gatherer' if $use_gath;
		print '<TH>Shepherd' if $use_shep;
		print '<TH>Search' if $use_search;
		print '<TH>Indexer' if $use_idx;
		print '<TH>MUX' if $use_mux;

		print '<TR>';
		print '<TD><INPUT type="checkbox" name="gath_all" value="x">All' if $use_gath;
		print '<TD><INPUT type="checkbox" name="shep_all" value="x">All' if $use_shep;
		print '<TD><INPUT type="checkbox" name="search_all" value="x">All' if $use_search;
		print '<TD><INPUT type="checkbox" name="idx_all" value="x">All' if $use_idx;
		print '<TD><INPUT type="checkbox" name="mux_all" value="x">All' . "\n" if $use_mux;

		my @gs = ($use_gath ? reverse print_stats("gath_stats",@gath_allstats) : ());
		my @hs = ($use_shep ? reverse print_stats("shep_stats",@shep_allstats) : ());
		my @is = ($use_idx ? reverse print_stats("idx_stats",@indexer_allstats) : ());
		my @ss = ($use_search ? reverse print_stats("search_stats",@search_allstats) : ());
		my @ms = ($use_mux ? reverse print_stats("mux_stats",@mux_allstats) : ());

		while (scalar(@gs+@is+@hs+@ss+@ms)>0) {
			my $g = pop @gs if $use_gath;
			my $h = pop @hs if $use_shep;
			my $i = pop @is if $use_idx;
			my $s = pop @ss if $use_search;
			my $m = pop @ms if $use_mux;
			print '<TR>';
			print '<TD>' . (defined $g ? $g : '') if $use_gath;
			print '<TD>' . (defined $h ? $h : '') if $use_shep;
			print '<TD>' . (defined $s ? $s : '') if $use_search;
			print '<TD>' . (defined $i ? $i : '') if $use_idx;
			print '<TD>' . (defined $m ? $m : '') . "\n" if $use_mux;
		}
		print "</TABLE>\n";

		if($use_search) {
			print "Search statistics for servers:<BR>\n";
			print qq(<INPUT type="checkbox" name="sp_all">All<BR>);
			for my $p (@search_prefixes) {
				print qq(<INPUT type="checkbox" name="search_pref" value="$p">$p\n);
			}
		}

		print <<END
<BR><INPUT type="checkbox" name="force" value="force">Force recomputing<BR>
<HR>
<INPUT type="submit" name="action" value="Examine">
<INPUT type="reset">
</FORM>
END
		;
		print '<P><A href="log/words/">See word statistics</A>' if $use_lex;
		print "<HR>\n";
	}

	if ($time_input eq "lastday") {
		my $t = time();
		$bdate = strftime("%Y-%m-%d",localtime($t-24*3600));
		$btime = strftime("%H-%M",localtime($t-24*3600));
		$edate = strftime("%Y-%m-%d",localtime($t));
		$etime = strftime("%H-%M",localtime($t));
	} elsif ($time_input eq "lastweek") {
		my $t = time();
		$bdate = strftime("%Y-%m-%d",localtime($t-7*24*3600));
		$btime = strftime("%H-%M",localtime($t-7*24*3600));
		$edate = strftime("%Y-%m-%d",localtime($t));
		$etime = strftime("%H-%M",localtime($t));
	} elsif ($time_input eq "lastmonth") {
		my $t = time();
		$bdate = strftime("%Y-%m-%d",localtime($t-30*24*3600));
		$btime = strftime("%H-%M",localtime($t-30*24*3600));
		$edate = strftime("%Y-%m-%d",localtime($t));
		$etime = strftime("%H-%M",localtime($t));
	} else {
		$btime =~ s/:/-/;
		$etime =~ s/:/-/;
	}

	@gath_stats = @gath_allstats if $gath_all;
	@shep_stats  = @shep_allstats if $shep_all;
	@search_stats = @search_allstats if $search_all;
	@idx_stats = @indexer_allstats if $idx_all;
	@mux_stats = @mux_allstats if $mux_all;
	@search_pref = @search_prefixes if $sp_all;

	if($use_gath) {
		print "<H3>Gatherer statistics</H3>\n" if @gath_stats>0;
		foreach my $g (@gath_stats) {
			my $url="$graph_cgi_url?name=$g&prefix=$gatherd_prefix&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
				.($force ? "&force=true" : "");
			print qq(<IMG src="$url" alt="There should be graph $g"><BR>\n);
		}
	}

	if($use_shep) {
		print "<H3>Shepherd statistics</H3>\n" if @shep_stats>0;
		foreach my $s (@shep_stats) {
			my $tmp_prefix = ($s eq $shep_stat_status) ? $shepherd_status_prefix : $shepherd_prefix;
			my $url="$graph_cgi_url?name=$s&prefix=$tmp_prefix&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
				.($force ? "&force=true" : "");
			print qq(<IMG src="$url" alt="There should be graph $s"><BR>\n);
		}
	}

	if($use_idx) {
		print "<H3>Indexer statistics</H3>\n" if @idx_stats>0;
		foreach my $i (@idx_stats) {
			my $tmp_prefix = ($i eq $idx_stat_status) ? $indexer_status_prefix : $indexer_prefix;
			my $url="$graph_cgi_url?name=$i&prefix=$tmp_prefix&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
				.($force ? "&force=true" : "");
			print qq(<IMG src="$url" alt="There should be graph $i"><BR>\n);
		}
	}

	if($use_search) {
		foreach my $sp (@search_pref) {
			print "<H3>Search statistics for $sp</H3>\n" if @search_stats>0;
			foreach my $s (@search_stats) {
				my $url="$graph_cgi_url?name=$s&prefix=$sp&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
					.($force ? "&force=true" : "");
				print qq(<IMG src="$url" alt="There should be graph $s"><BR>\n);
			}
		}
	}

	if($use_mux) {
		print "<H3>Mux statistics</H3>\n" if @mux_stats>0;
		foreach my $m (@mux_stats) {
			my $url="$graph_cgi_url?name=$m&prefix=$mux_prefix&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
				.($force ? "&force=true" : "");
			print qq(<IMG src="$url" alt="There should be graph $m"><BR>\n);
		}
	}
}

