/*
 *	Sherlock Search Engine -- Word Index
 *
 *	(c) 1997-2006 Martin Mares <mj@ucw.cz>
 */

#undef LOCAL_DEBUG

#include "sherlock/sherlock.h"
#include "ucw/unaligned.h"
#include "ucw/mempool.h"
#include "ucw/wildmatch.h"
#include "ucw/unicode.h"
#include "charset/unicat.h"
#include "search/sherlockd.h"
#include "search/lexicon.h"

#include <stdio.h>
#include <string.h>
#include <alloca.h>

#define IS_TRACING (current_query->debug & DEBUG_WORDS)
#define TRACE(msg...) do { if (IS_TRACING) add_reply(msg); } while(0)

/*** General functions ***/

int
word_contains_accents(byte *s)
{
  uns u;

  for(;;)
    {
      s = utf8_get(s, &u);
      if (!u)
	return 0;
      if (u != Uunaccent(u))
	return 1;
    }
}

/*
 *  We represent each phrase by a list of ph_word's (corresponding to words
 *  written in the phrase), each of them pointing to a single struct word
 *  (we cannot use struct word directly since they can occur multiple times
 *  in a single phrase).
 *
 *  When we expand a word, its variants are represented by a list of struct
 *  variant's connected to struct word.
 */

struct ph_word {
  cnode n;
  struct simple *simple;		/* Simple search component this came from */
  struct word *word;
  uns pos;				/* Relative position inside phrase */
  byte *unacc;				/* Unaccented version of the word */
  byte magic;				/* Has been generated by magic transformations of simples */
  byte prox_after;			/* Followed by "*" */
  byte w[0];				/* Accented version of the word */
};

typedef struct ph_word *word_id_t;
static struct simple *ph_current_simple;
static clist *ph_current_list;
static struct ph_word *ph_word_for_current_star;

void
word_dump_variants(struct word *w)
{
  SLIST_FOR_EACH(struct variant *, v, w->variants)
    {
      byte buf[MAX_WORD_BYTES+1];
      if (w->is_string)
	buf[0] = 0;
      else if (w->word_class == WC_COMPLEX)
	{
	  uns root, ctxt;
	  if (cplx_dissect_id(v->lex_id, &root, &ctxt))
	    sprintf(buf, "%04x (%04x)", root, ctxt);
	  else
	    sprintf(buf, "(%04x) %04x", ctxt, root);
	}
      else
	lex_extract(v->lex_id, buf);
      byte flg[9];
      strcpy(flg, "QSMAYLam");
      for (uns i=0; i<8; i++)
	if (!(v->flags & (1 << i)))
	  flg[i] = '-';
      add_reply(".X\t\t<%s> noacc=%d lmask=%08x pen=%d flags=%s chain=%llx+%x",
		buf, v->noaccent_only, v->lang_mask, v->penalty, flg,
		(long long)v->refchain_start, v->refchain_len);
    }
}

static void
word_dump_phrase(clist *phrase, char *comment)
{
  if (!IS_TRACING)
    return;

  add_reply(".X %s phrase:", comment);
  CLIST_FOR_EACH(struct ph_word *, p, *phrase)
    {
      struct word *w = p->word;
      add_reply(".X\tpos=%d idx=%d class=%d stat=%d wild=%d magic=%d prox=%d '%s' '%s'",
		p->pos, w->index, w->word_class, w->status, w->is_wild, p->magic,
		p->prox_after, p->w, p->unacc ? : (byte*)"");
      word_dump_variants(w);
    }
}

static inline int
word_variant_subset(struct variant *a, struct variant *b)
{
  return
    (a->lang_mask & b->lang_mask) == a->lang_mask &&
    a->noaccent_only >= b->noaccent_only &&
    a->penalty >= b->penalty;
}

static int
word_add_variant(struct word *w, uns lex_id, uns noaccent_only, u32 lang_mask, uns penalty, uns flags)
{
  struct variant *v, *x, *xpred;

  if (w->var_count >= max_word_matches)
    {
      w->status = 105;
      return 0;
    }
  if (w->word_class != WC_COMPLEX && w->word_class != lex_get(lex_id)->class)
    {
      DBG("word_add_variant: class clash, ignoring");
      return 1;
    }

  v = mp_alloc(current_query->results->pool, sizeof(struct variant));
  v->lex_id = lex_id;
  v->noaccent_only = noaccent_only;
  v->lang_mask = lang_mask;
  v->penalty = penalty;
  v->flags = flags;
  v->refchain_start = 0;
  v->refchain_len = 0;

  SLIST_WALK_DELSAFE(x, w->variants, xpred)
    if (x->lex_id == v->lex_id)
      {
	/* If one of the variants implies the other, merge them */
	if (word_variant_subset(v, x))
	  {
	    x->flags |= v->flags;
	    return 1;
	  }
	else if (word_variant_subset(x, v))
	  {
	    v->flags |= x->flags;
	    slist_remove_after(&w->variants, &xpred->n);
	    slist_insert_after(&w->variants, &v->n, &xpred->n);
	    return 1;
	  }
      }
    else if (x->lex_id > v->lex_id)
      break;

  w->var_count++;
  slist_insert_after(&w->variants, &v->n, &xpred->n);
  return 1;
}

void
word_extract_variant(struct word *w, struct variant *v, byte *buf)
{
  if (v->flags & VF_SYNTHETIC)
    {
      /* XXX: Here we assume that the only synthetic words ever generated
       * are query words not present in the lexicon. At this point, we cannot
       * take them from the fake lexicon slots, because they are not preserved
       * in cached query results.
       */
      strcpy(buf, w->word);
    }
  else
    lex_extract(v->lex_id, buf);
}

static uns
word_synthesize(uns idx, byte *wd, uns class)
{
  /* If the word is already present in the lexicon (which can happen if it's a context
   * word), we need to use its real lex_id, because sometimes we compare lex_ids.
   */
  int id = lex_find_exact(wd);
  if (id >= 0)
    return id;

  uns len = strlen(wd);
  struct lex_entry *l = mp_alloc_zero(current_query->pool, sizeof(struct lex_entry) + len);
  l->class = class;
  l->length = len;
  memcpy(l->w, wd, len);
  id = current_dbase->lexicon_words + idx;
  current_dbase->lex_array[id] = l;
  return id;
}

/*** Lexical mapping of phrases ***/

static uns
translate_accent_mode(struct ph_word *p, uns acc)
{
  switch (acc)
    {
    case ACCENT_AUTO:
      return (current_query->contains_accents ? ACCENT_AUTO : ACCENT_STRIP);
    case ACCENT_STRIP:
    case ACCENT_STRICT:
      return acc;
    case ACCENT_AUTO_LOCAL:
      return (strcmp(p->w, p->unacc) ? ACCENT_AUTO : ACCENT_STRIP);
    default:
      ASSERT(0);
    }
}

static enum word_class
lm_lookup(enum word_class orig_class, u16 *uni, uns ulen, word_id_t *thisw, void *user UNUSED)
{
  struct ph_word *p;
  byte wbuf[MAX_WORD_BYTES+1], *wp=wbuf;
  byte ubuf[MAX_WORD_BYTES+1], *up=ubuf;
  uns u, wl, ul, i;
  struct word *w;
  struct expr *expr = ph_current_simple->raw;

  if (!uni)
    {
      ph_word_for_current_star = NULL;
      return orig_class;
    }
  for (i=0; i<ulen; i++)
    {
      u = uni[i];
      u = Utolower(u);
      wp = utf8_put(wp, u);
      u = Uunaccent(u);
      up = utf8_put(up, u);
    }
  *wp = *up = 0;
  wl = wp - wbuf + 1;
  ul = up - ubuf + 1;
  p = mp_alloc_zero(current_query->pool, sizeof(struct ph_word) + wl + ul);
  memcpy(p->w, wbuf, wl);
  p->unacc = p->w + wl;
  memcpy(p->unacc, ubuf, ul);
  int old_acc = expr->u.match.o.accent_mode;
  expr->u.match.o.accent_mode = translate_accent_mode(p, old_acc);
  w = lookup_word(current_query, expr, p->w);
  p->word = w;
  expr->u.match.o.accent_mode = old_acc;
  p->simple = ph_current_simple;
  if (orig_class != WC_NORMAL)
    w->word_class = orig_class;
  else if (ulen == 1 && uni[0] == '*')
    {
      w->word_class = WC_IGNORED;
      w->use_count--;
      if (ph_word_for_current_star)
	ph_word_for_current_star->prox_after = 1;
    }
  else
    {
      int c = query_word_classify(current_dbase, p->w);
      if (c < 0)
	{
	  if (ulen < lex_min_len_ign)
	    c = WC_IGNORED;
	  else if (ulen < lex_min_len)
	    c = WC_GARBAGE;
	  else
	    c = WC_NORMAL;
	}
      w->word_class = c;
      if ((wildcard_asterisks && strchr(p->w, '*')) || (wildcard_qmarks && strchr(p->w, '?')))
	w->is_wild = 1;
      ph_word_for_current_star = p;
    }
  if (w->word_class == WC_NORMAL)
    w->cover_count++;
  else
    w->status = 116;
  p->pos = ~0U;
  *thisw = p;
  return w->word_class;
}

static void
lm_got_word(uns pos, uns cat UNUSED, word_id_t p, void *user UNUSED)
{
  if (p->pos != ~0U)
    clist_remove(&p->n);
  clist_add_tail(ph_current_list, &p->n);
  p->pos = pos;
}

static inline void
lm_fixup_pos(word_id_t w, uns pos)
{
  if (w->pos != ~0U)
    clist_remove(&w->n);
  w->pos = pos;
  clist_add_tail(ph_current_list, &w->n);
}

static void
lm_got_complex(uns pos, uns cat UNUSED, word_id_t root, word_id_t context, uns dir, void *user UNUSED)
{
  if (!dir)
    {
      word_id_t x = root;
      root = context;
      context = x;
      pos--;
    }
  lm_fixup_pos(root, pos);
  lm_fixup_pos(context, pos+1);
}

#define LM_SIMPLE
#define LM_SEARCH_WILDCARDS
#include "indexer/lexmap.h"

#ifdef CONFIG_LANG

#define QUERY_LANGS current_query->lang_set

/*** Second pass of accent expansion (after morphological or synonymic expansion) ***/

static void
word_expand_post_acc(struct ph_word *p, uns require_flags)
{
  struct variant *v;
  struct word *w = p->word;

  if (p->word->options.accent_mode == ACCENT_STRICT)
    return;

  SLIST_WALK(v, w->variants)
    if (!(v->flags & VF_ACCENTIFIED) && (v->flags & require_flags))
      {
	byte wa[MAX_WORD_BYTES+1], wu[MAX_WORD_BYTES+1], la[MAX_WORD_BYTES+1], lu[MAX_WORD_BYTES+1];
	v->flags |= VF_ACCENTIFIED;
	lex_extract(v->lex_id, wa);
	lex_extract_noacc(v->lex_id, wu);
	uns chars = utf8_strlen(wu);
	ASSERT(chars <= MAX_WORD_CHARS);
	for (uns idx = lex_find_first(chars, wu); idx < current_dbase->lex_by_len[chars+1]; idx++)
	  {
	    if (idx == v->lex_id)
	      continue;
	    lex_extract(idx, la);
	    lex_extract_noacc(idx, lu);
	    if (strcmp(lu, wu))
	      break;
	    uns nonacc_only = 0;
	    uns penalty = v->penalty;
	    switch (p->simple->raw->u.match.o.accent_mode)	/* Use the non-translated accent mode */
	      {
	      case ACCENT_STRIP:
		break;
	      case ACCENT_AUTO:
	      case ACCENT_AUTO_LOCAL:
		if (strcmp(la, lu))
		  continue;
		nonacc_only = 1;
		penalty += misaccent_penalty;
		break;
	      default:
		ASSERT(0);
	      }
	    ASSERT(lex_get(idx)->class == w->word_class);
	    if (!word_add_variant(w, idx, nonacc_only, v->lang_mask, penalty,
				  v->flags | VF_ACCENTS | VF_ACCENTIFIED))
	      return;
	  }
      }
}

/*** Morphologic expansion ***/

static struct mempool *stemmer_pool;

static void
word_expand_morph_variant(struct ph_word *p, struct variant *v)
{
  struct word *w = p->word;
  struct stem_block *sb;
  byte wa[MAX_WORD_BYTES+1];
  struct word_node *stem;
  clist *stems;
  clist *sb_list = &current_dbase->stem_block_list;
  int stem_id;
  u32 *exp;

  lex_extract(v->lex_id, wa);
  CLIST_WALK(sb, *sb_list)
    {
      u32 sb_lang_mask = sb->stemmer->lang_mask;
      if (!(sb_lang_mask & QUERY_LANGS))
	continue;
      if (!stemmer_pool)
	stemmer_pool = mp_new(4096);
      else
	mp_flush(stemmer_pool);
      struct word_node req = {
	.word_form = WORD_FORM_OTHER,
	.stem_form = WORD_FORM_LEMMA,
	.unaccented = (w->options.accent_mode == ACCENT_STRIP),
	.w = wa
      };
      if (stems = lang_stem(sb->stemmer, &req, stemmer_pool))
	CLIST_WALK(stem, *stems)
	  {
	    if ((stem_id = lex_find_exact(stem->w)) >= 0)
	      {
		TRACE(".M <%s> -> <%s> (langs %x)", wa, stem->w, sb_lang_mask);
		add_reply("l%s %s", stem->w, wa);
		if (stem_id != (int) v->lex_id && lex_get(stem_id)->class == w->word_class)
		  word_add_variant(p->word, stem_id, 0, v->lang_mask & sb_lang_mask,
				   v->penalty + stem_penalty, VF_MORPHED | VF_MORPH | VF_LEMMA);
		if (exp = stem_lookup_expansion(sb->array, sb->array_items, lex_export_id(stem_id)))
		  {
		    while (!(*++exp & 0x80000000))
		      {
			uns idp = lex_import_id(*exp);
			if (IS_TRACING)
			  {
			    byte ww[MAX_WORD_BYTES+1];
			    lex_extract(idp, ww);
			    add_reply(".M\t-> <%s>", ww);
			  }
			if (idp != v->lex_id && lex_get(idp)->class == w->word_class)
			  word_add_variant(w, idp, 0, v->lang_mask & sb_lang_mask,
					   v->penalty + morph_penalty, VF_MORPHED | VF_MORPH);
		      }
		  }
	      }
	    else
	      TRACE(".M <%s> -> <%s> (langs %x) unknown", wa, stem->w, sb_lang_mask);
	  }
    }
}

static void
word_expand_morph(struct ph_word *p)
{
  struct variant *v, *tmp;

  if (!p->word->options.morphing)
    return;

  SLIST_WALK_DELSAFE(v, p->word->variants, tmp)
    if (!(v->flags & VF_MORPHED) && !v->noaccent_only)
      {
	v->flags |= VF_MORPHED;
	word_expand_morph_variant(p, v);
      }
  if (p->word->options.morphing > 1)
    word_expand_post_acc(p, VF_MORPH);
}

/*** Synonymic expansion ***/

struct syn_var {
  uns id;
  u32 lmask;
  struct variant *orig;
};

static struct syn_var *syn_vars;
static uns syn_n;

static void
word_expand_syn_variant(struct variant *v)
{
  struct syn_block *s;

  CLIST_WALK(s, current_dbase->syn_block_list)
    {
      if (!(s->lang_mask & v->lang_mask))
	continue;
      u32 *invexp = stem_lookup_expansion(s->inverse_array, s->inverse_items, lex_export_id(v->lex_id));
      if (!invexp)
	continue;
      invexp++;
      while (!(*invexp & 0x80000000))
	{
	  uns class = *invexp++;
	  TRACE(".Y Found synonymic class %x", class);
	  u32 *exp = stem_lookup_expansion(s->direct_array, s->direct_items, class);
	  ASSERT(exp);
	  exp++;
	  while (!(*exp & 0x80000000))
	    {
	      uns syn = lex_import_id(*exp++);
	      if (syn != v->lex_id && syn_n < HARD_MAX_SYNONYMA)
		{
		  uns i;
		  for (i=0; i<syn_n && syn_vars[i].id != syn; i++)
		    ;
		  if (i < syn_n)
		    syn_vars[i].lmask |= s->lang_mask & v->lang_mask;
		  else
		    syn_vars[syn_n++] = (struct syn_var){
		      .id = syn,
		      .lmask = s->lang_mask & v->lang_mask,
		      .orig = v
		    };
		}
	    }
	}
    }
}

static void
word_expand_synonyma(struct ph_word *p)
{
  struct variant *v, *tmp;

  if (!p->word->options.synonyming)
    return;

  syn_n = 0;
  syn_vars = alloca(HARD_MAX_SYNONYMA * sizeof(struct syn_var));
  SLIST_WALK_DELSAFE(v, p->word->variants, tmp)
    if (!(v->flags & VF_SYNONYMUM) &&
	((v->flags & VF_LEMMA) || !(v->flags & VF_MORPH)) &&
	!v->noaccent_only)
      word_expand_syn_variant(v);
  if (!syn_n)
    return;

  for (uns i=0; i<syn_n; i++)
    {
      struct syn_var *r = &syn_vars[i];
      byte worig[MAX_WORD_BYTES+1], wsyn[MAX_WORD_BYTES+1];
      lex_extract(r->orig->lex_id, worig);
      lex_extract(r->id, wsyn);
      add_reply("Y%s %s %s %d", wsyn, worig, p->word->word, i);
      if (p->word->options.synonyming >= 2 && (p->word->options.syn_expand & (1ULL << i)))
	word_add_variant(p->word, r->id, 0, r->lmask, r->orig->penalty + synonymum_penalty, VF_SYNONYMUM);
    }
  if (p->word->options.synonyming >= 3)
    word_expand_post_acc(p, VF_SYNONYMUM);
  if (p->word->options.synonyming >= 4)
    word_expand_morph(p);
}

#else
static inline void word_expand_morph(struct ph_word *p UNUSED) { }
static inline void word_expand_synonyma(struct ph_word *p UNUSED) { }
#define QUERY_LANGS ~0U
#endif

/*** Expansion of words to variants ***/

void
word_ensure_variants(struct word *w)
{
  // Beware, strange words can enter this place, even too long ones which are kicked out before entering the ph_list
  if (slist_empty(&w->variants) && w->word_class != WC_COMPLEX && utf8_strlen(w->word) <= MAX_WORD_CHARS)
    word_add_variant(w, word_synthesize(w->index, w->word, w->word_class), 0, QUERY_LANGS, 0, VF_QUERY | VF_SYNTHETIC | VF_ACCENTIFIED);
}

static void
word_expand(struct ph_word *p)
{
  struct word *w = p->word;
  byte lex[MAX_WORD_BYTES+1], lexu[MAX_WORD_BYTES+1];
  uns chars, idx, nonacc_only, penalty;
  uns seen_exact = 0;
  uns seen_stripped = 0;

  chars = utf8_strlen(p->unacc);
  ASSERT(chars <= MAX_WORD_CHARS);
  for (idx = lex_find_first(chars, p->unacc); idx < current_dbase->lex_by_len[chars+1]; idx++)
    {
      lex_extract(idx, lex);
      lex_extract_noacc(idx, lexu);
      if (strcmp(lexu, p->unacc))
	break;
      int exact_match = !strcmp(lex, p->w);
      int stripped_match = !strcmp(lex, p->unacc);
      seen_exact += exact_match;
      seen_stripped += stripped_match;
      nonacc_only = penalty = 0;
      switch (w->options.accent_mode)
	{
	case ACCENT_STRIP:
	  break;
	case ACCENT_STRICT:
	  if (!exact_match)
	    {
	      DBG("= %d <%s> <%s> !strict", idx, lex, lexu);
	      continue;
	    }
	  break;
	case ACCENT_AUTO:
	  if (!exact_match)
	    {
	      if (strcmp(lex, lexu))
		{
		  DBG("= %d <%s> <%s> !misaccented", idx, lex, lexu);
		  continue;
		}
	      nonacc_only = 1;
	      penalty = misaccent_penalty;
	    }
	  break;
	default:
	  ASSERT(0);
	}
      DBG("= %d <%s> <%s> %d %s", idx, lex, lexu, lex_get(idx)->class, nonacc_only ? "[nonacc]" : "");
      ASSERT(lex_get(idx)->class == w->word_class);
      if (!word_add_variant(w, idx, nonacc_only, QUERY_LANGS, penalty, VF_QUERY | VF_ACCENTIFIED | (exact_match ? VF_ACCENTS : 0)))
	return;
    }
  if (!seen_exact)
    word_add_variant(w, word_synthesize(w->index, w->word, w->word_class), 0, QUERY_LANGS, 0, VF_QUERY | VF_SYNTHETIC | VF_ACCENTIFIED);
  if (!seen_stripped && w->options.accent_mode == ACCENT_STRIP && strcmp(p->w, p->unacc))
    word_add_variant(w, word_synthesize(w->index + HARD_MAX_WORDS, p->unacc, w->word_class), 0, QUERY_LANGS, 0, VF_QUERY | VF_ACCENTS | VF_SYNTHETIC | VF_ACCENTIFIED);
}

static void
word_expand_wild(struct ph_word *p)
{
  struct word *w = p->word;
  int first, last, idx;
  uns accented, old, nonacc_only, pxlen;
  static struct mempool *wild_pool;
  struct wildpatt *patt, *pattu;
  byte *px;
  uns ignore_count = 0;

  /* Find non-wildcard prefix */
  px = p->unacc;
  pxlen = 0;
  while (*px != '*' && *px != '?')
    {
      uns u;
      px = utf8_get(px, &u);
      pxlen++;
    }
  if (pxlen < min_wildcard_prefix_len)
    {
      w->status = 114;
      return;
    }

  /* Compile both accented and unaccented pattern */
  accented = strcmp(p->w, p->unacc);
  if (!wild_pool)
    wild_pool = mp_new(16384);
  else
    mp_flush(wild_pool);
  patt = wp_compile(p->w, wild_pool);
  if (accented)
    pattu = wp_compile(p->unacc, wild_pool);
  else
    pattu = patt;
  if (!patt || !pattu)
    {
      w->status = 111;
      return;
    }

  uns total_zone_len = 0;
  for (uns ilen=pxlen; ilen<=MAX_WORD_CHARS; ilen++)
    {
      /* For each length, find the lexicon zone */
      if (!pxlen)
	{
	  first = current_dbase->lex_by_len[ilen];
	  last = current_dbase->lex_by_len[ilen+1];
	}
      else
	{
	  old = *px;
	  *px = 0;
	  first = lex_find_first(ilen, p->unacc);
	  (*(px-1))++;			/* We know 0xff is not a valid character code */
	  last = lex_find_first(ilen, p->unacc);
	  (*(px-1))--;			/* We know 0xff is not a valid character code */
	  *px = old;
	}
      DBG("Zone for length %d: [%d,%d)", ilen, first, last);
      total_zone_len += last - first;
      if (total_zone_len > max_wildcard_zone)
	{
	  w->status = 113;
	  return;
	}

      /* Scan the zone */
      for (idx=first; idx < last; idx++)
	{
	  byte lex[MAX_WORD_BYTES+1], lexu[MAX_WORD_BYTES+1];

	  nonacc_only = 0;
	  lex_extract(idx, lex);
	  switch (w->options.accent_mode)
	    {
	    case ACCENT_STRIP:
	      lex_extract_noacc(idx, lexu);
	      if (!wp_match(pattu, lexu))
		{
		  DBG("* %d <%s> !strip", idx, lexu);
		  continue;
		}
	      break;
	    case ACCENT_STRICT:
	      if (!wp_match(patt, lex))
		{
		  DBG("* %d <%s> !strict", idx, lex);
		  continue;
		}
	      break;
	    case ACCENT_AUTO:
	      if (!wp_match(patt, lex))
		{
		  lex_extract_noacc(idx, lexu);
		  if (!wp_match(pattu, lexu))
		    {
		      DBG("* %d <%s> !auto", idx, lex);
		      continue;
		    }
		  if (strcmp(lex, lexu))
		    {
		      DBG("* %d <%s> <%s> !misaccented", idx, lex, lexu);
		      continue;
		    }
		  nonacc_only = 1;
		}
	      break;
	    default:
	      ASSERT(0);
	    }
#ifdef LOCAL_DEBUG
	  lex_extract_noacc(idx, lexu);
	  DBG("* %d <%s> <%s> %d %s", idx, lex, lexu, lex_get(idx)->class, nonacc_only ? "[nonacc]" : "");
#endif
	  if (lex_get(idx)->class == WC_NORMAL)
	    {
	      if (!word_add_variant(w, idx, nonacc_only, QUERY_LANGS, 0, VF_QUERY | VF_ACCENTIFIED))
		return;
	    }
	  else
	    ignore_count++;
	}
    }
  if (ignore_count && slist_empty(&w->variants))
    w->status = 116;
}

static void
word_expand_std(clist *phrase)
{
  struct ph_word *p;

  CLIST_WALK(p, *phrase)
    {
      struct word *w = p->word;
      if (!w->expanded)
	{
	  w->expanded = 1;
	  if (w->is_wild)
	    word_expand_wild(p);
	  else
	    {
	      word_expand(p);
	      word_expand_morph(p);
	      word_expand_synonyma(p);
	    }
	}
    }
}

/*** Processing of complexes ***/

#ifdef CONFIG_CONTEXTS

static inline uns
word_get_context(struct word *w, struct variant *v)
{
  if (w->word_class == WC_COMPLEX)
    {
      uns root, ctxt;
      cplx_dissect_id(v->lex_id, &root, &ctxt);
      return root;
    }
  else
    {
      struct lex_entry *l = lex_get(v->lex_id);
      return GET_CONTEXT(&l->ctxt);
    }
}

static void
word_expand_complex(struct word *w, struct ph_word *root, struct ph_word *ctxt, uns dir)
{
  struct word *rw = root->word, *cw = ctxt->word;
  struct variant *rv, *cv;

  SLIST_WALK(rv, root->word->variants)
    if (!(rv->flags & VF_SYNTHETIC))
      SLIST_WALK(cv, ctxt->word->variants)
	if (!(cv->flags & VF_SYNTHETIC))
	  {
	    uns noacc = rv->noaccent_only | cv->noaccent_only;
	    uns lmask = rv->lang_mask & cv->lang_mask;
	    if (!lmask)
	      continue;
	    uns rc = word_get_context(rw, rv);
	    uns cc = word_get_context(cw, cv);
	    uns id = cplx_make_id(rc, cc, dir);
	    word_add_variant(w, id, noacc, lmask, rv->penalty, 0);
	  }
}

static struct ph_word *
do_make_complex(struct ph_word *root, struct ph_word *ctxt, uns dir)
{
  struct ph_word *p;
  struct word *w;
  byte *z;

  byte *rw = root->w;
  byte *cw = (ctxt->word->root ? : ctxt->word)->word;
  uns tl = strlen(rw) + strlen(cw) + 4;
  p = mp_alloc_zero(current_query->pool, sizeof(struct ph_word) + tl);
  z = p->w;
  if (!dir)
    {
      *z++ = '(';
      while (*cw)
	*z++ = *cw++;
      *z++ = ')';
      *z++ = ' ';
    }
  while (*rw)
    *z++ = *rw++;
  if (dir)
    {
      *z++ = ' ';
      *z++ = '(';
      while (*cw)
	*z++ = *cw++;
      *z++ = ')';
    }
  *z = 0;
  p->pos = root->pos;
  p->simple = root->simple;
  p->unacc = NULL;
  p->prox_after = (dir ? ctxt : root)->prox_after;
  DBG("Searching for complex <%s>", p->w);
  p->word = w = lookup_word(current_query, p->simple->raw, p->w);
  if (!w->expanded)
    {
      w->expanded = 1;
      w->word_class = WC_COMPLEX;
      w->root = root->word;
      word_expand_complex(w, root, ctxt, dir);
    }
  return p;
}

static struct ph_word *
word_make_complex(struct ph_word *root, struct ph_word *ctxt, uns dir, uns do_magic)
{
  uns is_magic = 0;

  if (do_magic)
    {
      /* If we're in magic mode, we do the normal condensations since they are
       * needed for the near matcher and if magic_complexes is turned on, we mark
       * some of the complexes as magic, so that they will be inserted to the
       * simple by word_add_magic_complexes().
       */
      if (root->simple != ctxt->simple)
	{
	  if (!magic_complexes)
	    return NULL;

	  /* Combine matching senses. Rules:
	   * (1) Words have incompatible senses (+ vs. -) => throw away.
	   * (2) Use sense of the root [by setting p->simple]
	   */
	  int rs = root->simple->raw->u.match.sense;
	  int cs = ctxt->simple->raw->u.match.sense;
	  if (rs > 0 && cs < 0 || rs < 0 && cs > 0)
	    return NULL;

	  /* If the root part has been already covered, we don't need a magic complex.
	   * If it hasn't, let's go forth and decrease its use_count, so that it won't
	   * be reported as non-indexed.
	   */
	  if (root->word->cover_count)
	    return NULL;
	  root->word->use_count--;

	  is_magic = 1;
	}
    }
  else
    root->word->cover_count++;

  /* Look it up as a phrase and set the magic flag. Such a fragment can
   * arise multiple times in a single simple query, but we don't take
   * care of them, they will be merged properly by the boolean optimizer.
   */
  struct ph_word *p = do_make_complex(root, ctxt, dir);
  p->magic = is_magic;
  p->simple = root->simple;
  return p;
}

static void
word_find_complexes(clist *phrase, uns do_magic)
{
  struct ph_word *p, *r, *prev, *next, *new;

  CLIST_WALK_DELSAFE(p, *phrase, r)
    if (p->word->word_class == WC_CONTEXT)
      {
	prev = clist_prev(phrase, &p->n);
	next = clist_next(phrase, &p->n);
	new = NULL;
	if (next && (next->word->word_class == WC_NORMAL || !prev))
	  new = word_make_complex(p, next, 1, do_magic);
	if (!new && prev)
	  new = word_make_complex(p, prev, 0, do_magic);
	if (new)
	  {
	    clist_insert_before(&new->n, &p->n);
	    clist_remove(&p->n);
	    p->word->use_count--;
	  }
      }
    else
      ASSERT(p->word->word_class == WC_NORMAL);
}

#endif

/*** Transformation of query expressions for phrases ***/

static void
word_xform_expr(struct query *q, struct simple *s)
{
  struct expr *e = s->raw;
  struct expr *f;

  if (clist_empty(&s->phrase))		/* The phrase is empty */
    e = new_node(EX_IGNORE);
  else if (!clist_next(&s->phrase, clist_head(&s->phrase))) /* Just a single word */
    {
      struct ph_word *p = clist_head(&s->phrase);
      struct word *w = p->word;
      w->is_outer = 1;
      w->weight = MAX(w->weight, e->u.match.o.weight);
      e = new_node(EX_WORD);
      e->u.word.w = w;
    }
  else
    {
      struct ph_word *p;
      uns firstpos = 0;
      uns lastpos = ~0U;
      uns is_prox = 0;
      if (q->nphrases >= max_phrases)
	{
	  add_err("-103 Too many phrases");
	  eval_err(103);
	}
      struct phrase *phrase = mp_alloc_zero(q->results->pool, sizeof(*phrase));
      q->phrases[q->nphrases] = phrase;
      phrase->index = q->nphrases++;
      phrase->weight = e->u.match.o.weight;
      e = new_node(EX_PHRASE);
      e->u.phrase.p = phrase;
      CLIST_WALK(p, s->phrase)
	{
	  if (phrase->length >= MAX_PHRASE_LEN)
	    {
	      add_err("-110 Phrase too complex");
	      eval_err(110);
	    }
	  struct word *w = p->word;
	  phrase->word[phrase->length] = w->index;
	  if (!phrase->length)
	    firstpos = lastpos = p->pos;
	  phrase->relpos[phrase->length] = MIN(p->pos - lastpos, 4);
	  phrase->next_same_word[phrase->length] = phrase->word_to_idx[w->index];
	  phrase->word_to_idx[w->index] = phrase->length+1;
	  if (is_prox)
	    phrase->prox_map |= 1 << phrase->length;
	  lastpos = p->pos;
	  phrase->length++;
	  f = new_node(EX_WORD);
	  f->u.word.w = p->word;
	  e = new_op(EX_AND, e, f);
	  is_prox = p->prox_after;
	}
    }
  s->cooked = e;
}

/*** Association of reference chains to words ***/

static void
word_find_refs(clist *phrase)
{
  CLIST_FOR_EACH(struct ph_word *, p, *phrase)
    {
      struct word *w = p->word;
      if (w->expanded >= 2)
	continue;
      w->expanded = 2;
      SLIST_FOR_EACH(struct variant *, v, w->variants)
	{
	  struct lex_entry *l;
#ifdef CONFIG_CONTEXTS
	  if (w->word_class == WC_COMPLEX)
	    l = cplx_get(v->lex_id);
	  else
#endif
	    l = lex_get(v->lex_id);
	  v->refchain_start = GET_O(l->ref_pos);
	  v->refchain_len = GET_U16(l->ch_len) << 12;
	}
    }
}

/*** Analysis of a single word/phrase match ***/

static void
word_analyse(struct query *q, struct simple *s)
{
  ph_current_simple = s;
  ph_current_list = &s->phrase;
  ph_word_for_current_star = NULL;

  clist_init(&s->phrase);
  lm_doc_start(NULL);
  if (!lm_map_text(s->raw->u.match.word))
    {
      add_err("-115 Word too long");
      eval_err(115);
    }
  word_dump_phrase(&s->phrase, "Initial");
  word_expand_std(&s->phrase);
  word_dump_phrase(&s->phrase, "Expanded");
#ifdef CONFIG_CONTEXTS
  word_find_complexes(&s->phrase, 0);
  word_dump_phrase(&s->phrase, "Complexified");
#endif
  word_xform_expr(q, s);
  word_find_refs(&s->phrase);
}

/*** Magic transformations of simple queries ***/

#ifdef CONFIG_CONTEXTS
static void
word_add_magic_complexes(clist *magic, clist *simp)
{
  struct ph_word *p;

  CLIST_WALK(p, *magic)
    if (p->magic)
      {
	struct simple *s = mp_alloc_zero(current_query->pool, sizeof(struct simple));
	struct word *w = p->word;
	clist_add_tail(simp, &s->n);
	s->raw = p->simple->raw;
	s->cooked = new_node(EX_WORD);
	s->cooked->u.word.w = w;
	w->is_outer = 1;
	w->weight = MAX(w->weight, s->raw->u.match.o.weight);
	w->use_count++;
      }
}
#endif

static void
word_add_magic_near(clist *magic)
{
  struct query *q = current_query;
  struct ph_word *pw, *pwlast;
  struct phrase *p;
  uns cnt = 0;
  uns lastpos = 0;

  if (q->nnears >= max_nears)
    return;
  pwlast = NULL;
  CLIST_WALK(pw, *magic)
    if (pw->simple->raw->u.match.sense >= 0)
      {
	if ((!pwlast || pwlast->simple != pw->simple) && !pw->magic)
	  cnt++;
	pwlast = pw;
      }
  if (cnt < 2)
    return;

  DBG("Adding near match for %d words", cnt);
  p = mp_alloc_zero(q->results->pool, sizeof(*p));
  q->nears[q->nnears++] = p;
  CLIST_WALK(pw, *magic)
    if (pw->simple->raw->u.match.sense >= 0 && p->length < MAX_PHRASE_LEN)
    {
      uns i = p->length;
      struct word *w = pw->word;
      p->word[i] = w->index;
      p->relpos[i] = MIN(pw->pos - lastpos, 4);
      lastpos = pw->pos;
      p->next_same_word[p->length] = p->word_to_idx[w->index];
      p->word_to_idx[w->index] = p->length+1;
      p->length++;
      pw->word->use_count++;
    }
}

static void
word_add_magic_merges(clist *magic, clist *simp)
{
  struct ph_word *p;
  struct simple *s;
  struct expr *e;
  byte w[MAX_WORD_BYTES+1];
  uns len = 0;
  int wt = 0;
  uns cnt = 0;
  uns type_mask = magic_merge_classes << 16;

  /* Require all words positive, without wildcards and no strict accents and merge non-accented variants */
  CLIST_WALK(p, *magic)
    {
      struct expr *e = p->simple->raw;
      uns l1 = strlen(p->unacc);
      if (e->u.match.sense < 0 ||
	  p->word->is_wild ||
	  strcmp(p->w, p->unacc) && e->u.match.o.accent_mode == ACCENT_STRICT)
	return;
      if (len + l1 > MAX_WORD_BYTES)
	return;
      memcpy(w+len, p->unacc, l1);
      len += l1;
      wt = wt + e->u.match.o.weight;
      type_mask &= e->u.match.type_mask;
      cnt++;
    }
  if (cnt < 2 || wt <= 0 || !type_mask)
    return;
  w[len] = 0;

  if (utf8_strlen(w) > MAX_WORD_CHARS)
    return;

  TRACE(".U <%s>", w);

  e = new_node(EX_MATCH);
  e->u.match.o.weight = wt + magic_merge_bonus;
  e->u.match.o.accent_mode = ACCENT_STRICT;
  e->u.match.word = mp_strdup(current_query->pool, w);
  e->u.match.type_mask = type_mask;

  s = mp_alloc_zero(current_query->pool, sizeof(*s));
  s->raw = e;
  word_analyse(current_query, s);

  if (p = clist_head(&s->phrase))
    {
      p->word->hide_count++;
      clist_add_tail(simp, &s->n);
    }
}

static void
word_add_magic_keyphrases(clist *magic, clist *simp)
{
  struct ph_word *p;
  struct simple *s;
  struct expr *e;
  byte w[MAX_WORD_BYTES+1];
  uns len = 0;

  /* Require all words positive and without wildcards */
  CLIST_WALK(p, *magic)
    {
      struct expr *e = p->simple->raw;
      uns l1;
      if (e->u.match.sense < 0 ||
	  p->word->is_wild || !(e->u.match.type_mask & (magic_keyphrases_classes << 16)))
	return;
      l1 = strlen(p->w);
      if (len)
        w[len++] = ' ';
      if (len + l1 > MAX_WORD_BYTES)
	return;
      memcpy(w + len, p->w, l1);
      len += l1;
    }
  w[len] = 0;
  if (!len)
    return;

  if (utf8_strlen(w) > MAX_WORD_CHARS)
    return;

  TRACE(".U <%s>", w);

  e = new_node(EX_MATCH);
  e->u.match.o.weight = magic_keyphrases_bonus;
  e->u.match.o.accent_mode = ACCENT_STRICT;
  e->u.match.word = mp_strdup(current_query->pool, w);
  e->u.match.type_mask = magic_keyphrases_string_types;
  e->u.match.is_string = 1;

  s = mp_alloc_zero(current_query->pool, sizeof(*s));
  s->raw = e;
  s->cooked = string_analyse(current_query, e);
  clist_add_tail(simp, &s->n);
}

static void
word_apply_magic(clist *simp)
{
  struct simple *s;
  clist phrase;
  struct ph_word *p;

  clist_init(&phrase);
  ph_current_list = &phrase;
  lm_doc_start(NULL);
  CLIST_WALK(s, *simp)
    if (!s->raw->u.match.is_string)
      {
	ph_current_simple = s;
	ph_word_for_current_star = NULL;
	lm_map_text(s->raw->u.match.word);
      }
  word_dump_phrase(&phrase, "Initial magic");
  if (magic_merge_words)
    word_add_magic_merges(&phrase, simp);
  if (magic_keyphrases)
    word_add_magic_keyphrases(&phrase, simp);
  if (magic_complexes || magic_near)
    {
      word_expand_std(&phrase);
      word_dump_phrase(&phrase, "Expanded");
#ifdef CONFIG_CONTEXTS
      word_find_complexes(&phrase, 1);
      word_dump_phrase(&phrase, "Complexified");
#endif
      word_find_refs(&phrase);
#ifdef CONFIG_CONTEXTS
      if (magic_complexes)
	word_add_magic_complexes(&phrase, simp);
#endif
      if (magic_near)
	word_add_magic_near(&phrase);
    }
  CLIST_WALK(p, phrase)
    p->word->use_count--;
}

/*** Main entry point: analysis of simple queries ***/

void
word_analyse_simple(struct query *q, clist *l)
{
  struct simple *s;
  uns i, cnt = 0;

  if (!q->dbase->lex_array)
    {
      CLIST_WALK(s, *l)
	if (!s->raw->u.match.is_string)
	  s->cooked = new_node(EX_NONE);
      return;
    }

  /* Reset per-word flag for the words already known */
  for (i=0; i<q->nwords; i++)
    q->words[i]->cover_count = 0;

  /* Analyse individual words/phrases */
  CLIST_WALK(s, *l)
    if (!s->raw->u.match.is_string)
      {
	word_analyse(q, s);
	cnt++;
      }

  /* Scan the whole simple query as a phrase and perform all the magic tricks */
  if ((cnt >= 2 && (magic_complexes || magic_near || magic_merge_words)) ||
      (cnt >= 1 && magic_keyphrases))
    word_apply_magic(l);
}

/*** Initialization ***/

void
words_init(struct database *db)
{
  if (!(db->parts & DB_PART_WORDS))
    return;

  static uns words_inited;
  if (!words_inited++)
    lm_init();
  lexicon_init(db);
}
