/*
 *	Sherlock Filter Engine --- tree data structure
 *
 * 	This is not an ordinary header file, but an internal included source
 * 	file.  Define the following macros:
 * 	
 * 	NAME
 * 	NAME_
 * 	STRUCT_DECL
 * 	PTR_DECL
 * 	STRUCT_ATTR
 * 	FORMAT
 * 	FULL_INTERVAL
 * 	TREE_KEY_{ENDSTRING,ATOMIC}, TREE_NOCASE, TREE_ATOMIC_TYPE
 * 	FILTER_TYPE
 * 	RAW_ATTR
 * 	ICASE_UNSIGN_FLAG
 * 	DUMP_TITLE
 *
 *	(c) 2004--2005, Robert Spalek <robert@ucw.cz>
 */

struct NAME_(node) {
	struct filter_cmd *cmd;
	int dir;
	STRUCT_DECL;
};

static void
NAME_(dump_key)(struct fastbuf *fb, struct NAME_(node) *n)
{
	bprintf(fb, "key=%" FORMAT " ", n->STRUCT_ATTR);
}

static void
NAME_(dump_data)(struct fastbuf *fb, struct NAME_(node) *n)
{
	bprintf(fb, "dir=%d cmd=%p", n->dir, n->cmd);
}

#define	TREE_NODE	struct NAME_(node)
#define	TREE_PREFIX(x)	NAME_(x)
#define	TREE_WANT_SEARCH
#define	TREE_WANT_NEW
#define	TREE_WANT_DUMP
#define	TREE_WANT_ITERATOR
#define	TREE_WANT_BOUNDARY
#include "lib/redblack.h"

static PTR_DECL NAME_(full_int)[2] = { FULL_INTERVAL };

struct NAME_(tree) *
NAME_(tree_new)(struct mempool *mp, struct filter_cmd *case_cmd)
{
	ASSERT(case_cmd->op == SWITCH);
	uns count = 0;
	for (struct filter_case *curr = case_cmd->c.swit.cases; curr; curr=curr->next)
		if (can_be_in_tree(curr, FILTER_TYPE, ICASE_UNSIGN_FLAG, NULL))
			count++;
	if (count < filter_tree_limit)
		return NULL;

	struct NAME_(tree) *tree = mp_alloc(mp, sizeof(struct NAME_(tree)));
	NAME_(init)(tree);
	struct filter_case **cs = &case_cmd->c.swit.cases;
	while (*cs)
	{
		union filter_raw_value2 str[2];
		str[0].RAW_ATTR = NAME_(full_int)[0];
		str[1].RAW_ATTR = NAME_(full_int)[1];
		if (!can_be_in_tree(*cs, FILTER_TYPE, ICASE_UNSIGN_FLAG, str))
		{
			cs = &(*cs)->next;
			continue;
		}
		int cmp;
		struct NAME_(node) *s0, *s1;
		int c0, c1;
		if ((PTR_DECL) str[0].RAW_ATTR == NAME_(full_int)[0])
		{
			cmp = 2;
			s0 = NULL;
			s1 = NAME_(search)(tree, str[1].RAW_ATTR);
		}
		else if ((PTR_DECL) str[1].RAW_ATTR == NAME_(full_int)[1])
		{
			cmp = -2;
			s0 = NAME_(search)(tree, str[0].RAW_ATTR);
			s1 = NULL;
		}
		else
		{
			cmp = NAME_(cmp)(str[0].RAW_ATTR, str[1].RAW_ATTR);
			if (cmp > 0)
			{
				log(L_WARN, "Filter tree optimization: empty interval %" FORMAT "..%" FORMAT, str[0].RAW_ATTR, str[1].RAW_ATTR);
				cs = &(*cs)->next;
				continue;
			}
			cmp = cmp ? -1 : 0;
			s0 = NAME_(search)(tree, str[0].RAW_ATTR);
			s1 = NAME_(search)(tree, str[1].RAW_ATTR);
		}
		if (s0)
			c0 = NAME_(cmp)(str[0].RAW_ATTR, s0->STRUCT_ATTR);
		else
			s0 = NAME_(boundary)(tree, 0), c0 = -1;
		if (s1)
			c1 = NAME_(cmp)(str[1].RAW_ATTR, s1->STRUCT_ATTR);
		else
			s1 = NAME_(boundary)(tree, 1), c1 = +1;
		c0 = c0 < 0 ? -1 : c0 > 0 ? +1 : 0;
		c1 = c1 < 0 ? -1 : c1 > 0 ? +1 : 0;
		if (s0 != s1
		|| s0 && c0 != c1
		|| s0 && (!c0 || s0->dir && s0->dir * c0 < 0))
		{
			log(L_WARN, "Filter tree optimization: overlapping interval %"
				FORMAT "..%" FORMAT " with %" FORMAT "/%d,%d %" FORMAT "/%d,%d",
				str[0].RAW_ATTR, str[1].RAW_ATTR, s0->STRUCT_ATTR, s0->dir, c0, s1->STRUCT_ATTR, s1->dir, c1);
			cs = &(*cs)->next;
			continue;
		}
		if (cmp < 2)
		{
			s0 = NAME_(new)(tree, str[0].RAW_ATTR);
			s0->dir = cmp;
			s0->cmd = (*cs)->positive;
		}
		if (cmp == -1 || cmp == 2)
		{
			s1 = NAME_(new)(tree, str[1].RAW_ATTR);
			s1->dir = cmp == -1 ? 1 : 2;
			s1->cmd = (*cs)->positive;
		}
		*cs = (*cs)->next;
	}
	return tree;
}

struct filter_cmd *
NAME_(tree_search)(struct NAME_(tree) *tree, PTR_DECL val)
{
	struct NAME_(node) *n = NAME_(search)(tree, val);
	if (!n)
		return NULL;
	int cmp = NAME_(cmp)(val, n->STRUCT_ATTR);
	if (n->dir * cmp < 0 || !cmp)
		return n->cmd;
	else
		return NULL;
}

static void
NAME_(dump1)(struct fastbuf *fb, int dir, PTR_DECL v1, PTR_DECL v2)
{
	bputs(fb, "case ");
	byte *op1;
	if (dir == 2)
		op1 = "<=";
	else if (dir == -2)
		op1 = ">=";
	else if (!dir)
		op1 = "==";
	else
		op1 = "=#";
	bputs(fb, op1);
	if (ICASE_UNSIGN_FLAG)
		bputc(fb, op1[1]);
	bputc(fb, ' ');
	if (FILTER_TYPE == F_ET_STRING)
		bputc(fb, '"');
	bprintf(fb, "%" FORMAT, dir == 2 ? v2 : v1);
	if (FILTER_TYPE == F_ET_STRING)
		bputc(fb, '"');
	if (op1[1] == '#')
	{
		bputs(fb, " .. ");
		if (FILTER_TYPE == F_ET_STRING)
			bputc(fb, '"');
		bprintf(fb, "%" FORMAT, v2);
		if (FILTER_TYPE == F_ET_STRING)
			bputc(fb, '"');
	}
	bputs(fb, ":\n");
}

void 
NAME_(tree_dump)(struct fastbuf *fb, struct NAME_(tree) *tree, uns level)
{
	PTR_DECL v1 = NAME_(full_int)[0];
	PTR_DECL v2;
	int last_dir = -1;
	struct filter_cmd *last_cmd = NULL;
	if (!tree)
		return;
	filter_dump_spaces(fb, level);
	bprintf(fb, "# Binary search tree for " DUMP_TITLE "\n");
	TREE_FOR_ALL(NAME, tree, n)
	{
		last_dir = n->dir;
		last_cmd = n->cmd;
		if (!n->dir)
			v1 = v2 = n->STRUCT_ATTR;
		else if (n->dir < 0)
		{
			v1 = n->STRUCT_ATTR;
			continue;
		} else
			v2 = n->STRUCT_ATTR;
		filter_dump_spaces(fb, level);
		NAME_(dump1)(fb, n->dir, v1, v2);
		filter_dump_commands(fb, n->cmd, level+1);
	}
	TREE_END_FOR;
	if (last_dir == -2)
	{
		filter_dump_spaces(fb, level);
		NAME_(dump1)(fb, last_dir, v1, v1);
		filter_dump_commands(fb, last_cmd, level+1);
	}
	if (filter_trace > 0)
	{
		bputs(fb, "/*\n");
		NAME_(dump)(fb, tree);
		bputs(fb, "*/\n");
	}
}

#undef	NAME
#undef	NAME_
#undef	STRUCT_DECL
#undef	PTR_DECL
#undef	STRUCT_ATTR
#undef	FORMAT
#undef	FULL_INTERVAL
#undef	FILTER_TYPE
#undef	RAW_ATTR
#undef	ICASE_UNSIGN_FLAG
#undef	DUMP_TITLE
