/*
 *	Sherlock: Custom Parts of Configuration
 *
 *	(c) 2001--2005 Martin Mares <mj@ucw.cz>
 *	(c) 2004 Robert Spalek <robert@ucw.cz>
 *
 *	This software may be freely distributed and used according to the terms
 *	of the GNU Lesser General Public License.
 */

/* Versions */

#define SHERLOCK_VERSION_SUFFIX ""	/* String appended to version number to identify custom versions */
#define CUSTOM_INDEX_TYPE 0x01		/* A byte identifying our flavor of index format */
#define CUSTOM_INDEX_VERSION 0x01	/* Remember to increase after each change of custom parts of the index */

/* Structures we'll need in function parameters */

struct card_attr;
struct odes;

/* Word types (at most 7 of them + WT_RESERVED and WT_MAX) */

#define WT_RESERVED		0	/* Reserved word type */
#define WT_TEXT			1	/* Ordinary text */
#define WT_EMPH			2	/* Emphasized text */
#define WT_SMALL		3	/* Small font */
#define WT_SMALL_HEADING	4	/* Heading */
#define WT_BIG_HEADING		5	/* Larger heading */
#define WT_ALT			6	/* Alternate texts for graphical elements */
#define WT_MAX			7

/* Descriptive names used for user output */
#define WORD_TYPE_USER_NAMES			\
   "reserved", "text", "emph", "small", "hdr1", "hdr2", "alt", "word7"

/* Keywords for word type names */
#define WORD_TYPE_NAMES	       			\
	T(TEXT, 1 << WT_TEXT)			\
	T(EMPH, 1 << WT_EMPH)			\
	T(SMALL, 1 << WT_SMALL)			\
	T(HDR, (1 << WT_SMALL_HEADING) | (1 << WT_BIG_HEADING))  \
	T(HDR1, 1 << WT_SMALL_HEADING)		\
	T(HDR2, 1 << WT_BIG_HEADING)		\
	T(ALT, 1 << WT_ALT)			\

/*
 * These types are always matched with/without accents if accent mode is set to "auto",
 * regardless of accentedness of the current document.
 */
#define WORD_TYPES_AUTO_ACCENT_ALWAYS_STRICT 0
#define WORD_TYPES_AUTO_ACCENT_ALWAYS_STRIP 0

/* These types belong to all languages */
#define WORD_TYPES_ALL_LANGS 0

/* These types don't belong to any language, so neither language matching nor lemmatization affects them */
#define WORD_TYPES_NO_LANG 0

/* Meta information types (at most 16 of them + MT_MAX) */

#define MT_TITLE		0	/* Document title */
#define MT_KEYWORD		1	/* Keyword from the document */
#define MT_MISC			2	/* Unclassified metas */
#define MT_URL_KEYWD		3	/* Keywords extracted from URL */
#define MT_FILE			4	/* Part of file name */
#define MT_EXT			5	/* External texts (link texts) */
#define MT_MAX			6

#define META_TYPE_USER_NAMES			\
   "title", "keywd", "meta", "urlword", "file", "ext", "meta6", "meta7",	\
   "meta8", "meta9", "meta10", "meta11", "meta12", "meta13", "meta14", "meta15"

/* Keywords for meta type names */
#define META_TYPE_NAMES	       			\
	T(TITLE, 1 << MT_TITLE)			\
	T(KEYWD, 1 << MT_KEYWORD)		\
	T(META, 1 << MT_MISC)			\
	T(URLWORD, 1 << MT_URL_KEYWD)		\
	T(FILE, 1 << MT_FILE)			\
	T(EXT, 1 << MT_EXT)			\

#define META_TYPES_AUTO_ACCENT_ALWAYS_STRICT 0
#define META_TYPES_AUTO_ACCENT_ALWAYS_STRIP ((1 << MT_FILE) | (1 << MT_URL_KEYWD))
#define META_TYPES_ALL_LANGS (1 << MT_EXT)
#define META_TYPES_NO_LANG ((1 << MT_URL_KEYWD) | (1 << MT_FILE))

/* String types (at most 7 of them + ST_RESERVED and ST_MAX) */

#define ST_RESERVED		0	/* Reserved string type */
#define ST_URL			1	/* URL of the document */
#define ST_HOST			2	/* Host name */
#define ST_DOMAIN		3	/* Domain name */
#define ST_REF			4	/* URL reference */
#define ST_MAX			5

#define STRING_TYPE_USER_NAMES							\
   "reserved", "URL", "host", "domain", "ref", "link", "type6", "type7"

#define STRING_TYPE_NAMES			\
	T(URL, 1 << ST_URL)			\
	T(HOST, 1 << ST_HOST)			\
	T(DOMAIN, 1 << ST_DOMAIN)		\
	T(REF, 1 << ST_REF)			\
	T(LINK, 1 << ST_REF)

#define STRING_TYPES_URL ((1 << ST_URL) | (1 << ST_REF))
/* These must be indexed in lowercase form */
#define STRING_TYPES_CASE_INSENSITIVE ((1 << ST_HOST) | (1 << ST_DOMAIN))

static inline void
custom_index_strings(struct odes *o UNUSED, void (*f)(byte *text, uns type) UNUSED)
{
  /*
   * Call f for all strings you want to add to the index.
   * ST_URL, ST_HOST, ST_DOMAIN and ST_REF are indexed automatically
   * if they are defined.
   */
}

/*
 *  Definitions of custom attributes:
 *
 *  First of all, you need to define your own card_attr fields which will
 *  contain your attributes: CUSTOM_CARD_ATTRS lists them.
 *  Please order the attributes by decreasing size to get optimum padding.
 *
 *  Then define custom_create_attrs() which will get the object description
 *  and set your card_attr fields accordingly.
 *
 *  Finally, you have to define CUSTOM_ATTRS with matching rules:
 *
 *  INT_ATTR(id, keyword, get_func, parse_func) -- unsigned integer attribute
 *
 *  id		C identifier of the attribute
 *  keywd	search server keyword for the attribute
 *  int get_func(struct card_attr *ca)
 *		get attribute value from the card_attr
 *  byte *parse_func(u32 *dest, byte *value, uns intval)
 *		parse value in query (returns error message or NULL)
 *		for KEYWD = "string", it gets value="string", intval=0
 *		for KEYWD = num, it gets value=NULL, intval=num.
 *
 *  SMALL_SET_ATTR(id, keyword, get_func, parse_func)
 *    -- integers 0..31 with set matching
 *
 *  A good place for definitions of the functions is lib/custom.c.
 *
 *  You can also use LATE_INT_ATTR and LATE_SMALL_SET_ATTR instead
 *  which makes the attribute to be matched as late as possible, so that
 *  mismatched documents will still get caught by CUSTOM_EARLY_STATS,
 *  at the expense of slightly slower searching.
 */

/* No custom attributes defined yet */

#define CUSTOM_CARD_ATTRS
#define CUSTOM_ATTRS

/* The following function fills type_flags if CONFIG_FILETYPE and all custom attributes */

void custom_create_attrs(struct odes *odes, struct card_attr *ca);

/*
 *  Definition of custom statistics:
 *
 *  Here you can introduce your own statistics for the search server
 *  (e.g., you can count the number of documents matched for each value of a given custom attribute)
 *  by defining the following macros (optional):
 *
 *	CUSTOM_STAT_VARS	declarations of per-query variables which will appear in struct query
 *	CUSTOM_INIT_STATS(struct query *q)
 *				initialize these variables in a given struct query
 *	CUSTOM_LATE_STATS(struct query *q, struct card_attr *a)
 *				update variables when a document is matched
 *	CUSTOM_EARLY_STATS(q,a)	update variables when a document matches everything except
 *				possibly for the LATE_xxx attributes.
 *	CUSTOM_SHOW_STATS(struct query *q, void (*add)(char *fmt, ...))
 *				print the final results -- `add' is a printf-like function
 *				which you should call for adding a line to the search server
 *				reply (it will appear in the per-database block).
 *
 *  The same system is used for per-filetype statistics, which are either late or early,
 *  depending on the CONFIG_COUNT_ALL_FILETYPES switch; see sherlock/index.h for how is this done.
 */

/*
 *  Definition of custom matching rules:
 *
 *  You also can define matchers which are even more general than custom attributes
 *  and which can influence not only acceptance of documents, but also their weight.
 *  This is controlled by the following macros (each of them optional):
 *
 *	CUSTOM_MATCH_VARS	declarations of per-query variables which will appear in struct query.
 *				For each keyword you use below, you MUST define u32 <id>_value.
 *	CUSTOM_MATCH_INIT(struct query *q)
 *				initialize there variables in a given struct query
 *	CUSTOM_MATCH_PARSE	a list of parsing rules for custom keywords; for each keyword, include
 *				CUSTOM_MATCH_KWD(id, keyword, char (*parse)(struct query *q, enum custom_op op, byte *value, uns intval).
 *				Whenever KEYWORD <op> "value" is seen in the query, parse(q,value,0) is called
 *				and it's expected to return either NULL or an error message. For KEYWORD <op> number,
 *				parse(q,NULL,number) is called likewise.
 *	int CUSTOM_MATCH_CACHE_KEY(struct query *q, byte *buf)
 *				generate a unique textual representation of the custom attributes
 *				for use as a cache lookup key. Gets pointer to a buffer and returns the number
 *				of characters generated.
 *	int CUSTOM_MATCH(struct query *q, struct card_attr *ca, int Q)
 *				called to determine whetner a single card (given by its attributes) matches.
 *				Return 1 if accepted or 0 otherwise and possibly set the Q bonus for the card (initially 0).
 *				Also set all <id>_value fields [SORTBY <keyword> will be recognized and will
 *				use the _value field as the sort key.] Matching should have no side-effects,
 *				use CUSTOM_xxx_STATS if you need side-effects.
 *	CUSTOM_MATCH_SHOW(struct query *q, struct card_attr *, void (*add)(char *fmt, ...))
 *				print extra attributes to card description -- like CUSTOM_SHOW_STATS.
 */

/*
 *  Custom initialization (optional):
 *
 *	CUSTOM_INIT()		run after sherlockd initializes itself and before it calls any other CUSTOM_xxx macros.
 */
