/*
 *	Sherlock Gatherer -- Content Encoding and Type Identification
 *
 *	(c) 1997--2002 Martin Mares <mj@ucw.cz>
 *	(c) 2001--2004 Robert Spalek <robert@ucw.cz>
 */

#include "sherlock/sherlock.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "lib/conf.h"
#include "lib/chartype.h"
#include "lib/fastbuf.h"
#include "lib/url.h"

#include "gather/gather.h"

/* Structures */

struct intype_filter {			/* Input content type / content encoding filter */
	node n;
	byte *src, *dest;
};

struct encoding {			/* Encoding */
	node n;
	byte *encoding;
	byte *drop_suffix;
};

struct type_to_enc {			/* Content type => encoding conversion */
	node n;
	byte *type;
	byte *encoding;
};

enum cbmode { CBMODE_SUFFIX, CBMODE_WILDCARD, CBMODE_EXTENSION, CBMODE_BINARY, CBMODE_IS_ASCII, CBMODE_END };
enum cbresult { CBRES_DONT_KNOW, CBRES_TRUE, CBRES_FALSE, CBRES_END };

struct crystal_ball {			/* Magic rules for datomanthy */
	node n;
	uns order;			/* Order of the rule */
	enum cbmode mode;		/* Crystal ball type */
	byte *rule;			/* Name or contents pattern */
	uns rule_len;
	byte *type;			/* Resulting type */
};

/* Configuration */

static uns trace_contents;
#define TRACE(x,y...) do { if (trace_contents) log(L_DEBUG, x,##y); } while (0)
#define XTRACE(x,y...) do { if (trace_contents > 1) log(L_DEBUG, x,##y); } while (0)
#define XXTRACE(x,y...) do { if (trace_contents > 2) log(L_DEBUG, x,##y); } while (0)

static list inenc_filters, encodings, type_to_encs, intype_filters;
struct list crystal_balls;	/* contains only modes != CBMODE_EXTENSION */
static uns cb_order = 1;
static int ascii_fraction = 1000;

struct extension_record {
	uns order;
	byte *extension;	/* key */
	byte *content_type;
};
#define HASH_NODE struct extension_record
#define HASH_PREFIX(p) ext_##p
#define HASH_KEY_STRING extension
#define HASH_WANT_FIND
#define HASH_WANT_NEW
#define HASH_USE_POOL cfpool
#define HASH_NOCASE
#define HASH_CONSERVE_SPACE
#include "lib/hashtable.h"

struct content_type_record {
	byte *content_type;	/* key */
	struct crystal_ball *cb;
};
#define HASH_NODE struct content_type_record
#define HASH_PREFIX(p) ct_##p
#define HASH_KEY_STRING content_type
#define HASH_WANT_FIND
#define HASH_WANT_FIND_NEXT
#define HASH_WANT_NEW
#define HASH_USE_POOL cfpool
#define HASH_CONSERVE_SPACE
#include "lib/hashtable.h"

static byte *snerr = "Syntax error";

static inline byte *
canonicalize_content_type(byte *ct)
{
	if (!ct || !strcmp(ct, "UNKNOWN"))
		return NULL;
	else
		return ct;

}

static byte *
parse_inenc(struct cfitem *c UNUSED, byte *l)
{
	byte *w[2];
	struct intype_filter *f = cfg_malloc(sizeof(struct intype_filter));

	if (wordsplit(l, w, 2) != 2)
		return snerr;
	f->src = w[0];
	f->dest = w[1];
	add_tail(&inenc_filters, &f->n);
	return NULL;
}

static byte *
parse_encoding(struct cfitem *c UNUSED, byte *l)
{
	byte *w[2];
	struct encoding *f = cfg_malloc(sizeof(struct encoding));
	int z;

	z = wordsplit(l, w, 2);
	if (z != 1 && z != 2)
		return snerr;
	f->encoding = w[0];
	f->drop_suffix = (z == 2) ? w[1] : NULL;
	add_tail(&encodings, &f->n);
	return NULL;
}

static byte *
parse_dtenc(struct cfitem *c UNUSED, byte *l)
{
	byte *w[2];
	struct type_to_enc *f = cfg_malloc(sizeof(struct type_to_enc));

	if (wordsplit(l, w, 2) != 2)
		return snerr;
	f->type = w[0];
	f->encoding = w[1];
	add_tail(&type_to_encs, &f->n);
	return NULL;
}

static byte *
parse_intype(struct cfitem *c UNUSED, byte *l)
{
	byte *w[2];
	struct intype_filter *f = cfg_malloc(sizeof(struct intype_filter));

	if (wordsplit(l, w, 2) != 2)
		return snerr;
	f->src = w[0];
	f->dest = w[1];
	add_tail(&intype_filters, &f->n);
	return NULL;
}

static byte *
parse_filter(struct cfitem *c, byte *l)
{
	byte *w[2];
	struct crystal_ball *f = cfg_malloc(sizeof(struct crystal_ball));
	struct content_type_record *ctr;

	if (wordsplit(l, w, 2) != 2)
		return snerr;
	switch (c->name[0])
	{
		case 'S': f->mode = CBMODE_SUFFIX; break;
		case 'W': f->mode = CBMODE_WILDCARD; break;
		case 'E': f->mode = CBMODE_EXTENSION; break;
		case 'B': f->mode = CBMODE_BINARY; break;
		default: ASSERT(0);
	}
	if (f->mode == CBMODE_BINARY)
	{
		int z, b = 0;
		byte *c = w[0];
		byte *d;

		while (*c)			/* Estimate length of compiled pattern */
		{
			if (!Cdigit(*c))
				return snerr;
			while (Cdigit(*c))
				c++;
			if (*c++ != ':')
				return snerr;
			if (*c == '"')
			{
				c++;
				while (*c != '"')
				{
					if (!*c)
						return snerr;
					c++, b += 2;
				}
				c++;
			}
			else if (Cxdigit(*c))
			{
				while (Cxdigit(*c))
					c++, b++;
				if (b & 1)
					return snerr;
			}
			else
				return snerr;
			if (*c)
			{
				if (*c == ',')
					c++;
				else
					return snerr;
			}
		}

		d = f->rule = cfg_malloc(b + 1);	/* Compile the pattern */
		c = w[0];
		while (*c)
		{
			b = 0;
			while (*c != ':')
				b = 10*b + *c++ - '0';
			if (b < 0 || b > 254)
				return "Position out of range";
			if (*++c == '"')
			{
				c++;
				while (*c != '"')
				{
					*d++ = b++;
					*d++ = *c++;
				}
				c++;
			}
			else while (Cxdigit(*c))
			{
				int d0, d1;
				*d++ = b++;
				d0 = *c++;
				d1 = *c++;
				z = (Cxvalue(d0) << 4) | Cxvalue(d1);
				*d++ = z;
			}
			if (b > 254)
				return "Position out of range";
			if (*c)
				c++;
		}
		*d = 0xff;
		f->rule_len = d - f->rule;
	}
	else			/* CBMODE_WILDCARD || CBMODE_SUFFIX || CBMODE_EXTENSION */
	{
		f->rule = w[0];
		f->rule_len = strlen(f->rule);
	}
	f->type = w[1];
	f->order = cb_order;
	if (f->mode == CBMODE_EXTENSION)
	{
		struct extension_record *rec = ext_new(f->rule);
		rec->content_type = w[1];
		rec->order = cb_order;
	}
	else
		add_tail(&crystal_balls, &f->n);
	cb_order++;
	ctr = ct_new(f->type);
	ctr->cb = f;
	return NULL;
}

static byte *
parse_isascii(struct cfitem *c UNUSED, byte *l)
{
	struct crystal_ball *f = cfg_malloc(sizeof(struct crystal_ball));
	struct content_type_record *ctr;

	f->mode = CBMODE_IS_ASCII;
	f->rule = NULL;
	f->rule_len = 0;
	f->type = l;
	f->order = cb_order++;
	add_tail(&crystal_balls, &f->n);
	ctr = ct_new(f->type);
	ctr->cb = f;
	return NULL;
}

static struct cfitem content_config[] = {
	{ "Content",	CT_SECTION,	NULL },
	{ "Trace",	CT_INT,		&trace_contents },
	{ "InEnc",	CT_FUNCTION,	parse_inenc },
	{ "Encoding",	CT_FUNCTION,	parse_encoding },
	{ "TypeEnc",	CT_FUNCTION,	parse_dtenc },
	{ "InType",	CT_FUNCTION,	parse_intype },
	{ "Suffix",	CT_FUNCTION,	parse_filter },
	{ "WildCard",	CT_FUNCTION,	parse_filter },
	{ "Extension",	CT_FUNCTION,	parse_filter },
	{ "Bytes",	CT_FUNCTION,	parse_filter },
	{ "IsAscii",	CT_FUNCTION,	parse_isascii },
	{ "AsciiFraction",CT_INT,	&ascii_fraction },
	{ NULL,		CT_STOP,	NULL }
};

static void CONSTRUCTOR
content_init_config(void)
{
	cf_register(content_config);
	init_list(&inenc_filters);
	init_list(&encodings);
	init_list(&type_to_encs);
	init_list(&intype_filters);
	init_list(&crystal_balls);
	ext_init();
	ct_init();
}

/* Perform input filtering of content types and encodigs */

static byte *enc_m, *type_m, *rename_m;		/* Logging messages */

static int
perform_translation(struct list *filters, byte **type, byte *which)
{
	struct intype_filter *f;
	int i = 0;

	if (!*type)
		return 0;
	DO_FOR_ALL(f, *filters)
	{
		int res;
		i++;
		if (which[8] == 'T')
			res = match_ct_patt(f->src, *type);
		else if (which[8] == 'E')
			res = !strcasecmp(f->src, *type);
		else
			ASSERT(0);
		if (res)
		{
			XTRACE("Rule #%d: In filter: %s=`%s'", i, which, f->dest);
			if (!strcmp(f->dest, "ERROR"))
			{
				log(L_ERROR_R, "Suspicious %s: %s (rule %d)", which, *type, i);
				*type = NULL;
			}
			else
			{
				*type = canonicalize_content_type(f->dest);
			}
			return 1;
		}
	}
	return 0;
}

static int
maybe_encoding(byte **enc,byte **type)			/* Maybe our content type is just an encoding? */
{
	struct type_to_enc *f;
	int i = 0;

	if(*enc || !*type)
		return 0;
	DO_FOR_ALL(f, type_to_encs)
	{
		i++;
		if (match_ct_patt(f->type, *type))
		{
			XTRACE("Rule #%d: Content type %s -> Encoding %s", i, *type, f->encoding);
			*type = NULL;
			*enc = f->encoding;
			enc_m = "[from content-type] ";
			return 1;
		}
	}
	return 0;
}

static int
encoding_to_type(byte *enc,byte **type)			/* Inverse conversion for checking the Content-Type */
{
	struct type_to_enc *f;
	int i = 0;

	DO_FOR_ALL(f, type_to_encs)
	{
		i++;
		if (!strcasecmp(f->encoding, enc))
		{
			XTRACE("Rule #%d: Encoding %s -> Content type %s", i, enc, f->type);
			*type = f->type;
			return 1;
		}
	}
	XTRACE("Content-Encoding %s can not be converted to Content-Type", enc);
	*type = NULL;
	return 0;
}

/* Content encodings */

static inline void
chop_off_suffix(byte *base, byte *s)	/* Chop off filename suffix when decoding */
{
	int l = strlen(base);
	int k = strlen(s);

	if (l >= k && !strcmp(base + l - k, s))
	{
		base[l - k] = 0;
		XTRACE("Renaming to `%s' by cutting suffix `%s'", base, s);
		rename_m = " [suffix cut]";
	}
}

void
cut_inenc_suffix(byte *name, byte *enc)	/* Cut the optional file suffix and return the encoding number */
{
	struct encoding *e;

	if (!enc || !name)
		return;

	DO_FOR_ALL(e, encodings)
		if (!strcasecmp(e->encoding, enc))
		{
			if (e->drop_suffix)
				chop_off_suffix(name, e->drop_suffix);
			return;
		}
}

/* Content type vaticination */

#define	BUFSIZE		4096
#define	HEADSIZE	256

static struct fastbuf *dt_fb;
static byte dt_head[HEADSIZE];
static uns dt_size;
static byte *dt_name, *dt_type, *dt_enc;
static uns dt_name_len;

#define	MAX_GRADE	1000

static inline int
is_ascii(void)
{
	byte buf[BUFSIZE];
	uns pos = 0;
	uns non_ascii = 0;

	bseek(dt_fb, 0, SEEK_SET);
	while (1)
	{
		byte *k;
		uns l;
		if (! (l = bread(dt_fb, buf, BUFSIZE)) )
			break;
		for(k=buf; l; k++, l--)
			if (*k < 0x20 && !Cblank(*k))
			{
				if (non_ascii < 50)
					XXTRACE("Found non-ASCII character %x at position %d", *k, pos+k-buf);
				non_ascii++;
			}
		pos += k-buf;
	}
	if (non_ascii)
	{
		uns limit = (MAX_GRADE - ascii_fraction) * pos / MAX_GRADE;
		XXTRACE("Found %d non-ASCII characters among %d, limit is %d", non_ascii, pos, limit);
		if (non_ascii > limit)
			return 0;
	}
	return 1;
}

static inline int
match_binary_pattern(byte *r)
{
	uns k, l;

	while (*r != 0xff)
	{
		k = *r++;		/* Address */
		l = *r++;		/* Data */
		if (k >= dt_size || l != dt_head[k])
		{
			XXTRACE("Binary pattern mismatch at position %d: %x!=%x", k, dt_head[k], l);
			return 0;
		}
	}
	XXTRACE("Binary pattern matches");
	return 1;
}

/* Reset at the beginning of vaticinate():  */
static int plausible_name;		/* set to 0 iff the filename contains parameters (probably a script) */
static int ascii_result;		/* cache flag */

static enum cbresult
evaluate_crystal_ball(struct crystal_ball *f)
{
	switch (f->mode)
	{
		case CBMODE_BINARY:
			if (!dt_fb)
				return CBRES_DONT_KNOW;
			if (!match_binary_pattern(f->rule))
				return CBRES_FALSE;
			break;
		case CBMODE_SUFFIX:
			{
				int start;
				if (!dt_name || !plausible_name)
					return CBRES_DONT_KNOW;
				start = dt_name_len - f->rule_len;
				if (start < 0 || strcasecmp(f->rule, dt_name + start))
					return CBRES_FALSE;
				break;
			}
		case CBMODE_WILDCARD:
			if (!dt_name || !plausible_name)
				return CBRES_DONT_KNOW;
			if(!match_pattern_nocase(f->rule, dt_name))
				return CBRES_FALSE;
			break;
		case CBMODE_IS_ASCII:
			if (!dt_fb)
				return CBRES_DONT_KNOW;
			if (ascii_result < 0)
				ascii_result = is_ascii();
			if (!ascii_result)
				return CBRES_FALSE;
			break;
		case CBMODE_EXTENSION:
			/* This is an exception, it takes part in when called
			 * from verification branch of vaticinate().  */
			if (!plausible_name)
				return CBRES_DONT_KNOW;
			else
				return CBRES_FALSE;
		default:
			die("Unknown crystal ball #%d", f->mode);
	}
	return CBRES_TRUE;
}

static int
validate_content_type(uns order, byte *ct, byte **type)
{
	struct crystal_ball *g;
	int valid = 1, binary_res[CBRES_END];
	bzero(binary_res, sizeof(binary_res));
	DO_FOR_ALL(g, crystal_balls)
		if (!strcmp(ct, g->type))
		{
			if (g->mode ==  CBMODE_IS_ASCII
			&& evaluate_crystal_ball(g) == CBRES_FALSE)
			{
				valid = 0;
				break;
			}
			if (g->mode == CBMODE_BINARY)
				binary_res [ evaluate_crystal_ball(g) ]++;
		}
	if (binary_res[ CBRES_FALSE ] && !binary_res[ CBRES_TRUE ])
		valid = 0;
	XTRACE("Rule #%d: looks like Content-Type %s, %s", order, ct,
		valid ? "accepted" : "rejected");
	if (valid)
	{
		*type = canonicalize_content_type(ct);
		type_m = "[guessed] ";
	}
	return valid;
}

static int
vaticinate(byte **type)		/* Check the existing Content-Type or find the first matching one if *type==NULL */
{
	byte *dt_name_dot = NULL;
	plausible_name = ! strchr(dt_name, '?');
	if (plausible_name)
		dt_name_dot = strrchr(dt_name, '.');
	dt_name_len = strlen(dt_name);
	ascii_result = -1;

	if (!*type)
	{
		struct crystal_ball *f;
		struct extension_record *hash_rec;

		if (ext_table.hash_count > 0 && dt_name_dot)
			hash_rec = ext_find(dt_name_dot + 1);
		else
			hash_rec = NULL;
		DO_FOR_ALL(f, crystal_balls)
		{
			if (hash_rec && f->order > hash_rec->order)
			{
				/* The hash-table record must be processed in
				 * right order according to other tests.  */
				if (validate_content_type(hash_rec->order, hash_rec->content_type, type))
					return 1;
				hash_rec = NULL;
			}
			if (evaluate_crystal_ball(f) == CBRES_TRUE)
			{
				if (validate_content_type(f->order, f->type, type))
					return 1;
			}
		}
		if (hash_rec)
		{
			if (validate_content_type(hash_rec->order, hash_rec->content_type, type))
				return 1;
		}
		XTRACE("No rule to guess Content-Type");
		type_m = "[unrecognized] ";
		return 0;
	}
	else
	{
		int results[CBMODE_END][CBRES_END];	/* number of results[CBMODE_*][CBRESULT_*] */
		int ruleid[CBMODE_END];			/* dismatched ruleid[CBMODE_*] */
		struct content_type_record *ctr;
		int i;

		memset(results, 0, sizeof(results));
		for (ctr=ct_find(*type); ctr; ctr=ct_find_next(ctr))
		{
			struct crystal_ball *f = ctr->cb;
			int res = evaluate_crystal_ball(f);	/* CBMODE_EXTENSION rules are also included */
			results[f->mode][res] ++;
			if (res==CBRES_FALSE)
				ruleid[f->mode] = f->order;
		}
		if (ext_table.hash_count > 0 && dt_name_dot
		&& ext_find(dt_name_dot + 1))
		{
			results[CBMODE_EXTENSION][CBRES_TRUE]++;
			results[CBMODE_EXTENSION][CBRES_FALSE]--;
		}
		for (i=CBMODE_WILDCARD; i<=CBMODE_EXTENSION; i++)
		{
			int j;
			if (results[i][CBRES_FALSE] > 0
			&& !results[CBMODE_SUFFIX][CBRES_FALSE])
				ruleid[CBMODE_SUFFIX] = ruleid[i];
			for (j=0; j<CBRES_END; j++)
			{
				results[CBMODE_SUFFIX][j] += results[i][j];
				results[i][j] = 0;
			}
		}
		type_m = "[confirmed] ";
		if (results[CBMODE_SUFFIX][CBRES_FALSE]
		&& !results[CBMODE_SUFFIX][CBRES_TRUE])
		{
			XTRACE("Rule #%d: unknown suffix `%s' for Content-Type %s, but it does not matter", ruleid[CBMODE_SUFFIX], dt_name, *type);
			type_m = "[confirmed but suffix] ";
		}
		if (results[CBMODE_BINARY][CBRES_FALSE]
		&& !results[CBMODE_BINARY][CBRES_TRUE])
		{
			XTRACE("Rule #%d: Content-Type %s is not faithful -- all binary patterns are invalid", ruleid[CBMODE_BINARY], *type);
			type_m = "[rejected pattern] ";
			return 0;
		}
		if (results[CBMODE_IS_ASCII][CBRES_FALSE])
		{
			XTRACE("Rule #%d: Content-Type %s is not faithful -- not an ascii file", ruleid[CBMODE_IS_ASCII], *type);
			type_m = "[rejected ascii] ";
			return 0;
		}
		XTRACE("Content-Type %s is faithful", *type);
		return 1;
	}
}

/* Setting the initial values of Content-Encoding and Content-Type.  */

void
set_content_encoding(byte *ce)
{
	gthis->content_encoding = ce;
	if (perform_translation(&inenc_filters, &ce, "Content-Encoding"))
	{
		TRACE("Content-Encoding translated from %s into %s",
			gthis->content_encoding ? : (byte*)"?",
			ce ? : (byte*)"?");
		gthis->content_encoding = ce;
	}
}

void
set_content_type(byte *ct)
{
	gthis->content_type = ct;
	if (perform_translation(&intype_filters, &ct, "Content-Type"))
	{
		TRACE("Content-Type translated from %s into %s",
			gthis->content_type ? : (byte*)"?",
			ct ? : (byte*)"?");
		gthis->content_type = ct;
	}
}

/* Parsing of the content type field */

void
parse_content_type(byte *l, byte **charset)
{
	byte *x, *y;

	/* Split the Content-Type value to real content type and charset info */

	while (*l && !Cspace(*l) && *l != ';')
		l++;
	while (*l)
	{
		if (Cspace(*l) || *l == ';')
		{
			*l++ = 0;
			continue;
		}
		x = l;
		while (*l && !Cspace(*l) && *l != ';' && *l != '=')
			l++;
		if (*l != '=')
			continue;
		*l++ = 0;
		y = l;
		if (*l == '"')
		{
			y++, l++;
			while (*l && *l != '"')
				if (*l == '\\' && l[1] == '"')
					l += 2;
				else
					l++;
			if (!*l)
				break;
			*l++ = 0;
		}
		else
		{
			while (*l && !Cspace(*l) && *l != ';')
				l++;
			if (*l)
				*l++ = 0;
		}
		/* x is the key, y is the value */
		if (!strcasecmp(x, "charset"))
			*charset = y;
	}
}

/* DATA TYPE PROCESSING -- MAIN ENTRY POINT */

static void
do_guess_content(void)
{
	byte *enc = dt_enc;
	byte *type = dt_type;
	int count;

	for (count=0; count<2; count++)			/* Run twice, because C-E gzip may be recognised from C-T app/gzip in 2nd pass */
	{
		byte namebuf[MAX_URL_SIZE], *name = NULL;
		uns rejected = 0;
		enc_m = type_m = rename_m = "";

		if (count==1 && !maybe_encoding(&enc, &type))	/* Content-Type actually converted to Content-Encoding */
			break;
		if (enc)					/* Virtually decode encoded files */
		{
			byte *orig_type = type;
			if (!encoding_to_type(enc, &type))	/* Trust unsupported Content-Encoding */
			{
				XTRACE("Unsupported Content-Encoding=%s with Content-Type=%s", enc, orig_type ? : (byte*) "?");
				type = NULL;
				enc_m = "[unsupported but trusted] ";
			}
			if (type && !vaticinate(&type))		/* We can confirm that it isn't properly encoded */
			{
				enc = type = NULL;
				enc_m = "[rejected] ";
			}
			else					/* Check the encoded contents */
			{
				type = orig_type;
				name = namebuf;
				strcpy(name, dt_name);
				cut_inenc_suffix(dt_name,enc);	/* Rename the file and hide the contents */
				bclose(dt_fb);
				dt_fb = NULL;
			}
			/* Fall-thru into checking the Content-Type */
		}
		if (!type || !vaticinate(&type))		/* Ignore non-faithful Content-Type */
		{
			if (type)
			{
				type = NULL;
				rejected = 1;
			}
			vaticinate(&type);			/* Find the best candidate */
		}

		TRACE("Contents: encoding %s%s->%s, type %s%s%s->%s%s",
			enc_m,
			dt_enc ? : (byte*) "?",
			enc ? : (byte*) "?",
			(byte*) (rejected ? "[rejected] " : ""),
			type_m,
			dt_type ? : (byte*) "?",
			type ? : (byte*) "?",
			rename_m);
		if (name)
			strcpy(dt_name, name);
		dt_enc = enc;
		dt_type = type;
	}

	bclose(dt_fb);
}

void
guess_content(void)		/* Modify content_{encoding,type} depending on all known attributes */
{
	dt_name = gthis->file_name ? : gthis->url_s.rest;
	dt_enc = gthis->content_encoding;
	dt_type = gthis->content_type;
	if (gthis->contents)				/* Read the head to check the binary patterns */
	{
		dt_fb = fbmem_clone_read(gthis->contents);
		dt_size = bread(dt_fb, dt_head, HEADSIZE);
		XXTRACE("Read %d bytes from head of the file", dt_size);
	}
	else
		dt_fb = NULL;
	do_guess_content();
	gthis->content_encoding = dt_enc;
	gthis->content_type = dt_type;
}

void
guess_content_by_name(byte *name, byte **type, byte **enc)
{
	dt_name = name;
	dt_enc = NULL;
	dt_type = NULL;
	dt_fb = NULL;
	do_guess_content();
	*type = dt_type;
	*enc = dt_enc;
}
