/*
 *	Sherlock Gatherer: Document Validator
 *
 *	(c) 2002 Martin Mares <mj@ucw.cz>
 */

#include "sherlock/sherlock.h"
#include "lib/conf.h"
#include "gather/gather.h"

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>

struct validator {
  node n;
  byte *ctype_patt;
  byte *command;
};

static list validator_list;
static uns validate_switch;

static byte *
add_validator(struct cfitem *item UNUSED, byte *arg)
{
  byte *w[2];
  struct validator *val = cfg_malloc(sizeof(struct validator));

  if (wordsplit(arg, w, 2) != 2)
    return "Expecting type mask and validation command";
  val->ctype_patt = w[0];
  val->command = w[1];
  add_tail(&validator_list, &val->n);
  return NULL;
}

static struct cfitem validate_config[] = {
  { "Validate",		CT_SECTION,	NULL },
  { "Validate",		CT_INT,		&validate_switch },
  { "Validator",	CT_FUNCTION, 	add_validator },
  { NULL,		CT_STOP,	NULL }
};

static void CONSTRUCTOR validate_init_config(void)
{
  cf_register(validate_config);
  init_list(&validator_list);
}

static void
validate_child(byte *cmd, int srcfd, int resfd)
{
  close(0);
  close(1);
  close(2);
  dup(srcfd);
  dup(resfd);
  dup(resfd);
  close(srcfd);
  close(resfd);
  execlp(cmd, cmd, NULL);
}

static void
validate_parent(int srcfd, int resfd)
{
  fd_set rset, wset;
  struct fastbuf *in = fbmem_clone_read(gthis->contents);
  struct fastbuf *out = fbmem_create(4096);
  int nfd = MAX(srcfd, resfd) + 1;
  int len;
  byte *buf;
  byte line[1024];

  gthis->temp = out;
  fcntl(srcfd, F_SETFL, fcntl(srcfd, F_GETFL, 0) | O_NONBLOCK);
  fcntl(resfd, F_SETFL, fcntl(resfd, F_GETFL, 0) | O_NONBLOCK);
  FD_ZERO(&rset);
  FD_ZERO(&wset);
  for(;;)
    {
      if (srcfd >= 0)
	FD_SET(srcfd, &wset);
      FD_SET(resfd, &rset);
      if (select(nfd, &rset, &wset, NULL, NULL) < 0)
	die("select: %m");
      if (srcfd >= 0 && FD_ISSET(srcfd, &wset))
	{
	  len = bdirect_read_prepare(in, &buf);
	  if (len > 0)
	    {
	      len = write(srcfd, buf, len);
	      if (len < 0)
		gerror(2601, "Error writing validator input: %m");
	      bdirect_read_commit(in, buf+len);
	    }
	  else
	    {
	      FD_CLR(srcfd, &wset);
	      close(srcfd);
	      srcfd = -1;
	    }
	}
      if (FD_ISSET(resfd, &rset))
	{
	  len = bdirect_write_prepare(out, &buf);
	  len = read(resfd, buf, len);
	  if (len < 0)
	    gerror(2601, "Error reading validator output: %m");
	  if (!len)
	    break;
	  bdirect_write_commit(out, buf+len);
	}
    }
  if (srcfd >= 0)
    close(srcfd);
  close(resfd);
  bclose(in);
  bflush(out);
  in = fbmem_clone_read(out);
  struct oattr *oa = NULL;
  while (bgets(in, line, sizeof(line)))
    oa = obj_add_attr(gthis->aa, 'j', line);
  if (!oa)
    obj_add_attr(gthis->aa, 'j', "# OK");
  bclose(in);
  bclose(out);
  gthis->temp = NULL;
}

static void
do_validate(struct validator *val)
{
  pid_t child, ch;
  int status, pipesrc[2], piperes[2];

  if (pipe(pipesrc) || pipe(piperes))
    die("pipe: %m");
  child = fork();
  if (child < 0)
    die("fork: %m");
  else if (child)
    {
      close(pipesrc[0]);
      close(piperes[1]);
      validate_parent(pipesrc[1], piperes[0]);
      ch = wait(&status);
      if (ch != child)
	die("wait: received pid %d instead of %d", ch, child);
      byte err[EXIT_STATUS_MSG_SIZE];
      if (format_exit_status(err, status))
	gerror(2600, "Validator %s", err);
    }
  else
    {
      close(pipesrc[1]);
      close(piperes[0]);
      validate_child(val->command, pipesrc[0], piperes[1]);
    }
}

void
validate_document(void)
{
  struct validator *val;

  if (!validate_switch)
    return;
  WALK_LIST(val, validator_list)
    if (match_ct_patt(val->ctype_patt, gthis->content_type))
      {
	do_validate(val);
	break;
      }
}
