/*
 *	Sherlock Search Engine -- Configuration
 *
 *	(c) 1997--2004 Martin Mares <mj@ucw.cz>
 */

#include "sherlock/sherlock.h"
#include "lib/conf.h"
#include "lib/chartype.h"
#include "lib/ipaccess.h"
#include "lib/unicode.h"
#include "search/sherlockd.h"

#include <string.h>
#include <stdlib.h>

byte *log_name;
byte *status_name;
uns log_incoming;
uns log_rejected;
uns log_requests;
uns log_replies;
uns port = 4444;
uns listen_queue = 16;
uns connection_timeout = 60;
byte *control_password = "";
struct database *databases;
uns num_matches = 100;
uns cache_size = 1;
uns max_output_matches = ~0;
uns max_words;
uns max_phrases;
uns max_nears;
uns max_bools;
uns max_word_matches;
uns global_accent_mode;
uns max_wildcard_zone = ~0;
uns min_wildcard_prefix_len;
uns global_allow_approx;
uns global_context_chars;
uns global_title_chars;
uns global_intervals;
uns global_site_max;
uns global_url_max = 0x7fffffff;
uns global_redirect_url_max = 0x7fffffff;
uns global_partial_answers;
uns global_morphing;
uns global_spelling;
uns global_synonyming;
uns global_syn_expand;
uns doc_weight_scale = 1;
uns word_weight_scale = 1;
uns word_bonus;
uns default_word_types = 0xffff00ff;
uns global_debug;
uns mem_map_zone_size = 16;
uns mem_map_elide_gaps = 16384;
uns mem_map_prefetch = 1;
uns prox_penalty;
uns prox_limit;
uns query_watchdog;
uns global_sorting;
uns global_sort_reverse;
uns magic_complexes;
uns magic_merge_words;
uns magic_merge_classes;
uns magic_near;
uns near_bonus_word;
uns near_penalty_gap;
uns near_bonus_connect;
uns misaccent_penalty;
uns blind_match_penalty;
uns morph_penalty;
uns stem_penalty;
uns synonymum_penalty;
uns spell_good_freq;
uns spell_min_len;
uns spell_margin;
uns spell_add_penalty;
uns spell_del_penalty;
uns spell_mod_penalty;
uns spell_xpos_penalty;
uns spell_accent_penalty;
uns spell_common_penalty;
uns spell_dwarf;
uns spell_dwarf_margin;
struct spell_pair *spell_common_pairs;
uns second_best_reduce = 1;
uns magic_merge_bonus;
struct ipaccess_list *access_list;
uns hydra_processes;

static struct database *config_this_db;

static byte *
cf_access(struct cfitem *c, byte *b)
{
  return ipaccess_parse(access_list, b, (c->name[0] == 'A'));
}

static byte *
cf_database(struct cfitem *a UNUSED, byte *c)
{
  struct database *d = cfg_malloc(sizeof(struct database));
  struct database *f;
  byte *fields[5];

  if (wordsplit(c, fields, 5) != 5)
    return "Invalid database definition";
  for (f=databases; f; f=f->next)
    if (!strcmp(f->name, fields[0]))
      return "Database of this name already defined";
  if (config_this_db)
    config_this_db->next = d;
  else
    databases = d;
  bzero(d, sizeof(*d));
  config_this_db = d;
  d->name = fields[0];
  d->fn_params = fields[1];
  d->fn_cards = fields[2];
  d->fn_card_attrs = fields[3];
  d->fn_references = fields[4];
  return NULL;
}

static byte *
cf_word_idx(struct cfitem *a UNUSED, byte *c)
{
  byte *fields[2];
  int x;

  if (!config_this_db)
    return "No current database";
  if ((x = wordsplit(c, fields, 2)) != 1 && x != 2)
    return "Invalid word index definition";
  config_this_db->fn_lexicon = fields[0];
  config_this_db->fn_stems = (x == 2) ? fields[1] : NULL;
  return NULL;
}

static byte *
cf_string_idx(struct cfitem *a UNUSED, byte *c)
{
  byte *fields[2];

  if (!config_this_db)
    return "No current database";
  if (wordsplit(c, fields, 2) != 2)
    return "Invalid string index definition";
  config_this_db->fn_string_map = fields[0];
  config_this_db->fn_string_hash = fields[1];
  return NULL;
}

static byte *
cf_weights(struct cfitem *a, byte *c)
{
  byte *fields[8];

  if (!config_this_db)
    return "No current database";
  if (wordsplit(c, fields, 8) != 8)
    return "Invalid weight definition";
  for (int i=0; i<8; i++)
    {
      uns w = atol(fields[i]);
      if (w >= 255)
	return "Weights are limited to 0..255";
      (a->name[0]=='W' ? config_this_db->word_weights : config_this_db->string_weights)[i] = w;
    }
  return NULL;
}

static byte *
cf_meta_weights(struct cfitem *a UNUSED, byte *c)
{
  byte *fields[16];

  if (!config_this_db)
    return "No current database";
  if (wordsplit(c, fields, 16) != 16)
    return "Invalid weight definition";
  for (uns i=0; i<16; i++)
    {
      uns w = 0;
      uns j = 0;
      uns ww = 0;
      byte *x = fields[i];
      while (*x)
	{
	  if (j > 3)
	    return "A single meta type can have at most 4 weights";
	  ww = atol(x);
	  while (*x && *x != '/')
	    x++;
	  if (*x)
	    x++;
	  if (ww > 255)
	    return "Weights are limited to 0..255";
	  w |= (ww << 8*j++);
	}
      while (j < 4)
	w |= (ww << 8*j++);
      config_this_db->meta_weights[i] = w;
    }
  return NULL;
}

static byte *
cf_card_prints(struct cfitem *a UNUSED, byte *c)
{
  if (!config_this_db)
    return "No current database";
  config_this_db->fn_card_prints = c;
  return NULL;
}

static byte *
cf_optional(struct cfitem *a UNUSED, byte *c)
{
  if (!config_this_db)
    return "No current database";
  if (c[0] >= '0' && c[0] <= '1' && !c[1])
    {
      config_this_db->is_optional = c[0] - '0';
      return NULL;
    }
  return "IsOptional must be either 0 or 1";
}

static byte *
cf_default_sort_by(struct cfitem *a UNUSED, byte *c)
{
  int x;

  if (*c == '-')
    {
      global_sort_reverse = ~0U;
      c++;
    }
  else
    global_sort_reverse = 0;
  if ((x = lookup_custom_attr(c)) < 0)
    return "Unknown attribute";
  global_sorting = x;
  return NULL;
}

static byte *
cf_spell_pair(struct cfitem *a UNUSED, byte *c)
{
  while (*c)
    {
      if (Cspace(*c))
	c++;
      else
	{
	  struct spell_pair *p = cfg_malloc(sizeof(*p));
	  GET_UTF8(c, p->x);
	  if (!*c || Cspace(*c))
	    return "Letter pairs expected";
	  GET_UTF8(c, p->y);
	  if (*c && !Cspace(*c))
	    return "Letter pairs expected";
	  p->next = spell_common_pairs;
	  spell_common_pairs = p;
	}
    }
  return NULL;
}

static struct cfitem ss_config[] = {
  { "Search",		CT_SECTION,	NULL },
  { "LogFile",		CT_STRING,	&log_name },
  { "StatusFile",	CT_STRING,	&status_name },
  { "LogIncoming",	CT_INT,		&log_incoming },
  { "LogRejected",	CT_INT,		&log_rejected },
  { "LogRequests",	CT_INT,		&log_requests },
  { "LogReplies",	CT_INT,		&log_replies },
  { "Port",		CT_INT,		&port },
  { "ListenQueue",	CT_INT,		&listen_queue },
  { "Allow",		CT_FUNCTION,	cf_access },
  { "Deny",		CT_FUNCTION,	cf_access },
  { "ConnTimeOut",	CT_INT,		&connection_timeout },
  { "ControlPassword",	CT_STRING,	&control_password },
  { "Database",		CT_FUNCTION,	cf_database },
  { "WordIndex",	CT_FUNCTION,	cf_word_idx },
  { "StringIndex",	CT_FUNCTION,	cf_string_idx },
  { "WordWeights",	CT_FUNCTION,	cf_weights },
  { "MetaWeights",	CT_FUNCTION,	cf_meta_weights },
  { "StringWeights",	CT_FUNCTION,	cf_weights },
  { "CardPrints",	CT_FUNCTION,	cf_card_prints },
  { "IsOptional",	CT_FUNCTION,	cf_optional },
  { "CacheSize",	CT_INT,		&cache_size },
  { "NumMatches",	CT_INT,		&num_matches },
  { "MaxOutputObjects",	CT_INT,		&max_output_matches },
  { "MaxWords",		CT_INT,		&max_words },
  { "MaxPhrases",	CT_INT,		&max_phrases },
  { "MaxNears",		CT_INT,		&max_nears },
  { "MaxBools",		CT_INT,		&max_bools },
  { "MaxWordMatches",	CT_INT,		&max_word_matches },
  { "AccentMode",	CT_INT,		&global_accent_mode },
  { "MaxWildcardZone",	CT_INT,		&max_wildcard_zone },
  { "MinWildcardPrefix",CT_INT,		&min_wildcard_prefix_len },
  { "AllowApprox",	CT_INT,		&global_allow_approx },
  { "DocWeightScale",	CT_INT,		&doc_weight_scale },
  { "WordWeightScale",	CT_INT,		&word_weight_scale },
  { "ContextChars",	CT_INT,		&global_context_chars },
  { "TitleChars",	CT_INT,		&global_title_chars },
  { "Intervals",	CT_INT,		&global_intervals },
  { "SiteMax",		CT_INT,		&global_site_max },
  { "URLMax",		CT_INT,		&global_url_max },
  { "RedirectURLMax",	CT_INT,		&global_redirect_url_max },
  { "PartialAnswers",	CT_INT,		&global_partial_answers },
  { "DefaultWordTypes",	CT_INT,		&default_word_types },
  { "Debug",		CT_INT,		&global_debug },
  { "WordBonus",	CT_INT,		&word_bonus },
  { "MemMapZone",	CT_INT,		&mem_map_zone_size },
  { "MemMapElideGaps",	CT_INT,		&mem_map_elide_gaps },
  { "MemMapPrefetch",	CT_INT,		&mem_map_prefetch },
  { "ProxPenalty",	CT_INT,		&prox_penalty },
  { "ProxLimit",	CT_INT,		&prox_limit },
  { "QueryWatchdog",	CT_INT,		&query_watchdog },
  { "DefaultSortBy",	CT_FUNCTION,	cf_default_sort_by },
  { "MagicComplexes",	CT_INT,		&magic_complexes },
  { "MagicMergeWords",	CT_INT,		&magic_merge_words },
  { "MagicMergeClasses",CT_INT,		&magic_merge_classes },
  { "MagicNear",	CT_INT,		&magic_near },
  { "NearBonusWord",	CT_INT,		&near_bonus_word },
  { "NearPenaltyGap",	CT_INT,		&near_penalty_gap },
  { "NearBonusConnect", CT_INT,		&near_bonus_connect },
  { "BlindMatchPenalty",CT_INT,		&blind_match_penalty },
  { "MisaccentPenalty",	CT_INT,		&misaccent_penalty },
  { "StemPenalty",	CT_INT,		&stem_penalty },
  { "MorphPenalty",	CT_INT,		&morph_penalty },
  { "Morphing",		CT_INT,		&global_morphing },
  { "Spelling",		CT_INT,		&global_spelling },
  { "SpellGoodFreq",	CT_INT,		&spell_good_freq },
  { "SpellMinLen",	CT_INT,		&spell_min_len },
  { "SpellMargin",	CT_INT,		&spell_margin },
  { "SpellAddPenalty",	CT_INT,		&spell_add_penalty },
  { "SpellDelPenalty",	CT_INT,		&spell_del_penalty },
  { "SpellModPenalty",	CT_INT,		&spell_mod_penalty },
  { "SpellXposPenalty",	CT_INT,		&spell_xpos_penalty },
  { "SpellAccentPenalty",	CT_INT,		&spell_accent_penalty },
  { "SpellCommonPenalty",	CT_INT,		&spell_common_penalty },
  { "SpellDwarf",	CT_INT,		&spell_dwarf },
  { "SpellDwarfMargin",	CT_INT,		&spell_dwarf_margin },
  { "SynonymumPenalty",	CT_INT,		&synonymum_penalty },
  { "Synonyming",	CT_INT,		&global_synonyming },
  { "SynExpand",	CT_INT,		&global_syn_expand },
  { "SecondBestReduce",	CT_INT,		&second_best_reduce },
  { "SpellCommonPairs",	CT_FUNCTION,	cf_spell_pair },
  { "MagicMergeBonus",	CT_INT,		&magic_merge_bonus },
  { "HydraProcesses",	CT_INT,		&hydra_processes },
  { NULL,		CT_STOP,	NULL }
};

static void CONSTRUCTOR ss_conf_init(void)
{
  access_list = ipaccess_init();
  cf_register(ss_config);
}
