/*
 *	Sherlock Search Engine -- Databases
 *
 *	(c) 1997--2004 Martin Mares <mj@ucw.cz>
 */

#include "sherlock/sherlock.h"
#include "lib/lfs.h"
#include "sherlock/index.h"
#include "lib/fastbuf.h"
#include "indexer/lexicon.h"
#include "indexer/params.h"
#include "search/sherlockd.h"

#include <string.h>
#include <alloca.h>
#include <unistd.h>
#include <sys/mman.h>

struct lexicon_config lexicon_config;

void
db_switch_config(struct database *db)
{
  memcpy(&lexicon_config, &db->params->lex_config, sizeof(struct lexicon_config));
}

struct merge_status {
  struct merge_status *next;
  struct card_attr *attrs;
  struct fastbuf *fb;
  struct card_print next_fp;
};

static void
db_reset_attrs(struct database *db)
{
  struct card_attr *attrs = db->card_attrs;
  for (uns i=0; i<db->num_ids; i++)
    attrs[i].flags &= ~CARD_FLAG_OVERRIDEN;
}

static void
db_merge(void)
{
  struct merge_status *m, *mm, **mp, *first_ms = NULL;
  uns index_cnt = 0;
  uns override_cnt = 0;

  for (struct database *db=databases; db; db=db->next)
    if (db->fb_card_prints)
      {
	m = alloca(sizeof(*m));
	m->attrs = db->card_attrs;
	m->fb = db->fb_card_prints;
	if (breadb(m->fb, &m->next_fp, sizeof(struct card_print)))
	  {
	    m->next = first_ms;
	    first_ms = m;
	  }
	index_cnt++;
      }

  for(;;)
    {
      int eq = 0;
      mm = first_ms;
      if (unlikely(!mm))
	break;
      m = mm->next;
      if (unlikely(!m))
	break;
      while (m)
	{
	  int cmp = memcmp(&m->next_fp.fp, &mm->next_fp.fp, sizeof(struct fingerprint));
	  if (cmp < 0)
	    {
	      mm = m;
	      eq = 0;
	    }
	  else if (!cmp)
	    eq++;
	  m = m->next;
	}
      if (eq)
	{
	  mp = &mm->next;
	  while (m = *mp)
	    {
	      if (!memcmp(&m->next_fp.fp, &mm->next_fp.fp, sizeof(struct fingerprint)))
		{
		  m->attrs[m->next_fp.cardid].flags |= CARD_FLAG_OVERRIDEN;
		  override_cnt++;
		  if (unlikely(!breadb(m->fb, &m->next_fp, sizeof(struct card_print))))
		    {
		      *mp = m->next;
		      continue;
		    }
		}
	      mp = &m->next;
	    }
	}
      if (unlikely(!breadb(mm->fb, &mm->next_fp, sizeof(struct card_print))))
	{
	  for (mp=&first_ms; (m = *mp) != mm; mp = &m->next)
	    ;
	  *mp = mm->next;
	}
    }

  for (struct database *db=databases; db; db=db->next)
    if (db->fb_card_prints)
      {
	bclose(db->fb_card_prints);
	db->fb_card_prints = NULL;
	msync(db->card_attrs, db->num_ids * sizeof(struct card_attr), MS_SYNC);
	if (mprotect(db->card_attrs, db->num_ids * sizeof(struct card_attr), PROT_READ) < 0)
	  die("Cannot reprotect card attributes read-only: %m");
      }

  log(L_INFO, "Merged %d indices: %d cards overriden", index_cnt, override_cnt);
}

void
db_init(void)
{
  struct database *db;
  uns size;
  int fd;
  int seen_prints = 0;

  for (db=databases; db; db=db->next)
    {
      fd = open(db->fn_params, O_RDONLY);
      if (fd < 0)
	{
	  if (db->is_optional)
	    {
	      log(L_INFO, "Database %s missing", db->name);
	      continue;
	    }
	  die("Cannot open %s: %m", db->fn_params);
	}
      db->params = xmalloc(sizeof(struct index_params) + 1);
      int e = read(fd, db->params, sizeof(struct index_params) + 1);
      if (e < 0)
	die("Cannot read database parameters from %s: %m", db->fn_params);
      if (e != sizeof(struct index_params) || db->params->version != INDEX_VERSION)
	die("%s: Incompatible index", db->fn_params);
      close(fd);
      db_switch_config(db);
      if (db->fn_card_prints)
	{
	  db->fb_card_prints = bopen(db->fn_card_prints, O_RDONLY, 65536);
	  seen_prints++;
	  db->card_attrs = mmap_file(db->fn_card_attrs, &size, 1);
	}
      else
	db->card_attrs = mmap_file(db->fn_card_attrs, &size, 0);
      db->num_ids = size / sizeof(struct card_attr);
      if (db->num_ids)
	db->num_ids--;
      db->card_attrs_end = db->card_attrs + db->num_ids;
      db->fd_cards = sh_open(db->fn_cards, O_RDONLY);
      if (db->fd_cards < 0)
	die("Unable to open %s: %m", db->fn_cards);
      db->card_file_size = sh_seek(db->fd_cards, 0, SEEK_END);
      db->fd_refs = sh_open(db->fn_references, O_RDONLY);
      if (db->fd_refs < 0)
	die("Unable to open %s: %m", db->fn_references);
      db->ref_file_size = sh_seek(db->fd_refs, 0, SEEK_END);
      log(L_INFO, "Loaded database %s: %d documents", db->name, db->num_ids);
      if (db->fn_card_prints)
	db_reset_attrs(db);
      words_init(db);
      strings_init(db);
    }
  if (seen_prints)
    db_merge();
}

struct database *
attr_to_db(struct card_attr *attr, oid_t *ooid)
{
  for (struct database *db=databases; db; db=db->next)
    if (attr >= db->card_attrs && attr < db->card_attrs_end)
      {
	if (ooid)
	  *ooid = attr - db->card_attrs;
	return db;
      }
  die("attr_to_db: Orphan object with attribute %p", attr);
}
