#!/usr/bin/perl -w

use Getopt::Std;
use IO::Socket;
use IO::Handle;
require 'syscall.ph';

# constants
$fork_forward = 2;

# set default values of params
$host = "localhost";
$port = 8192;
$filter = "";
$pat = $subst = "";
$rate = -1; # infinite
$count = -1; # infinite
$maxtime = -1; # infinite
$logfilename = "tmp/bench-$$.log";

########################################################################
######################## M A I N #######################################
########################################################################

# parse command-line
if(!getopts('h:x:s:f:r:n:t:'))
{
  die "Usage: $0 [switches] [filename]\n",
      "switches:\n",
      "-h <host>[:<port>]  use sherlockd at <host>, <port>\n",
      "-x <regex>          use only requests matching <regex>\n",
      "-s <pat>/<str>      substitute pattern <pat> with string <str> in requests\n",
      "-f [1|2|p]          use only phrases, one (two)-word requests\n",
      "-r <rate>           request rate (1/sec)\n",
      "-n <count>          run only <count> requests\n",
      "-t <sec>            run max <sec> seconds\n",
      "[filename]          list of input requests, if not provided stdin is used\n";
}

# override default params by params provided on command-line
$filter = $opt_x if defined $opt_x;
$rate = $opt_r if defined $opt_r;
$count = $opt_n if defined $opt_n;
$maxtime = $opt_t if defined $opt_t;

if(defined $opt_h)
{
  $host = $opt_h;
  @x = split(':', $opt_h);
  if(@x > 1) { ($host, $port) = @x; }
}

if(defined $opt_f)
{
  if($opt_f eq "1") # one word query
  {
    $filter = '^[^"]*"[^" ]+"[^"]*$';
  }
  elsif($opt_f eq "2") # two words query
  {
    $filter = '^[^"]*"[^" ]+"[^"]+"[^" ]+"[^"]*$';
  }
  elsif($opt_f eq "p") # phrase query
  {
    $filter = '"[^" ]+ [^" ]+"';
  }
}

if(defined $opt_s)
{
  @x = split('/', $opt_s);
  if(@x > 1) { ($pat, $subst) = @x; }
}

# print out all params
print
  "---------------------------------\n",
  "Host: $host\n",
  "Port: $port\n",
  "Filter: ",($filter?$filter:"(none)")."\n",
  "Substitution: ",($pat?"$pat -> $subst":"(none)")."\n",
  "Request-rate: ",($rate!=-1?$rate:"(maximum)")," req/s\n",
  "# of requests: ",($count!=-1?$count:"(whole input)"),"\n",
  "Max time: ",($maxtime!=-1?$maxtime:"(no limit)"),"\n",
  "---------------------------------\n";

# initialize counters, timers, ...
$period = ($rate!=-1) ? 1000/$rate : 0; # milisecons between requests
$n_req = 0;
$sleep = 0;
$connect_errors = 0;
die("Error: Cannot create log file!") if !open(LOG, ">$logfilename");

# we are using syscall gettimeofday for precise scheduling of requests
$tv = pack("LL", 0, 0);
$tz = pack("ll", 0, 0);
($sec, $usec) = get_time();
$time_offset = $sec;
$startmtime = $last_msec = int($usec/1000);

# read input requests
while( $line = <> )
{
  chomp($line);

  last if($count != -1 && $n_req >= $count);

  # filter out requests by $filter regex
  next if($filter ne "" && $line !~ /$filter/);

  $line =~ s/$pat/$subst/ if ($pat ne "");
  
  # spawn background task
  run_task($line);

  # wait for child (but not for this just spwned)
  if($n_req >= $fork_forward)
  {
    $ret = wait();
    if($ret > 0 && $? != 0) { $connect_errors++; }
    if($connect_errors > 3) { sleep 1; die "Error: Too many connect errors!"; }
  }
  $n_req++;

  # get precise time
  ($sec, $usec) = get_time();
  
  $msec = 1000*($sec-$time_offset)+int($usec/1000);
  $mdelta = $msec-$last_msec;
  $last_msec = $msec;

  # sleep if needed
  if($period > 0)
  {
    $sleep = $sleep+($period-$mdelta);
    if($sleep > 0) { select(undef, undef, undef, $sleep/1000); } # sleep
  }
  $real_time = ($sec-$time_offset)+($usec/1000000-$startmtime/1000);

  last if($maxtime != -1 && $real_time >= $maxtime);

  printf "\r$n_req\tspeed: %.3f req/s     \t",$n_req/$real_time;
  STDOUT->flush;
}

# wait for rest of children
for($i = 0; $i <= $fork_forward; $i++) { wait; }

# compute latency from LOG
close LOG;
die("Error: Cannot re-open log file!") if !open(LOG, $logfilename);
$latency_sum = 0;
while (<LOG>) { $latency_sum += $_; }
close(LOG);
unlink($logfilename);

# print summary
printf
  "\n".
  "---------------------------------\n".
  "SUMMARY:\n".
  "req. processed: %d\n".
  "time elapsed: %.3f s\n".
  "average speed: %.3f req/s\n".
  "average latency: %d ms\n".
  "---------------------------------\n",
  $n_req, $real_time, $n_req/$real_time, $latency_sum/$n_req;

exit 0;

########################################################################
sub get_time
{
  # get current seconds & microseconds
  syscall(&SYS_gettimeofday, $tv, $tz);
  return unpack("LL", $tv);
}

########################################################################
sub run_task
{
  my($req)= @_;
  local $/;
  $pid = fork();
  if(!defined $pid) { die "fork() failed!"; }
  if($pid == 0) # child
  {
    # get time before connect
    ($sec1, $usec1) = get_time();

    $sd = IO::Socket::INET->new(Proto=>"tcp", PeerAddr=>$host,PeerPort=>$port);
    if(!$sd) { $!= 1; die "Cannot connect to $host:$port!"; }
    print $sd $req."\n";
    $/ = "\n";
    $line = <$sd>;
    if(substr($line, 0, 1) ne "+") { print $line; }
    undef $/;
    while($line = <$sd>) {}
    #print $req."\n";

    # get time after reading whole reply
    ($sec2, $usec2) = get_time();

    $msec = ($sec2-$sec1)*1000 + ($usec2-$usec1)/1000;
    printf LOG "%d\n", $msec;

    exit 0; # end process
  }
}
