/*
 *	A Simple Utility for Fast Transfer of Large Files: Sender
 *
 *	(c) 2002 Martin Mares <mj@ucw.cz>
 *
 *	The file transfer protocol is really simple: on the sender side, you
 *	run 'ssh <target> "cd <destdir>; file-recv <dir> <files>" | file-send <dir>'.
 *	For each file, file-recv sends to its stdout a request line:
 *	"#FS# <ipaddr> <port> <secret-key> <filename>\n" and file-send responds
 *	by connecting to <port> at <ipaddr>, sending header followed by file data
 *	followed by MD5 signature of <secret-key> and Adler checksum of the file.
 *	File names must not contain slashes.
 */

#include "sherlock/sherlock.h"
#include "lib/conf.h"
#include "lib/lfs.h"
#include "lib/md5.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <zlib.h>
#include <sys/socket.h>
#include <sys/mman.h>
#include <sys/user.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>

static uns trace = 1;
static uns compression = 0;

/* Smallish chunks are more friendly to caches */
#define CHUNK_SIZE 65536

static struct cfitem filesend_config[] = {
  { "FileSend",		CT_SECTION,	NULL },
  { "Trace",		CT_INT,		&trace },
  { "Compression",	CT_INT,		&compression },
  { NULL,		CT_STOP,	NULL }
};

static void
swrite(int sk, void *buf, int len)
{
  byte *b = buf;
  int l;

  while (len > 0)
    {
      l = write(sk, b, len);
      if (l < 0)
	die("write: %m");
      if (!l)
	die("write: 0 bytes written -- why?");
      len -= l;
      b += l;
    }
}

static void
process_request(byte *hostname, byte *hostport, byte *key, byte *file)
{
  struct hostent *he;
  struct sockaddr_in sa;
  int sk, fd;
  s64 size, xfer_size = 0;
  sh_off_t where = 0;
  u32 compress_flag = compression;
  struct MD5Context md5c;
  byte md5[16];
  uns time;
  u32 adler;

  init_timer();

  if (strchr(file, '/'))
    die("Invalid file name `%s' received", file);
  if (!(he = gethostbyname(hostname)))
    die("Unable to resolve %s: %m", hostname);
  bzero(&sa, sizeof(sa));
  sa.sin_family = AF_INET;
  sa.sin_port = htons(atol(hostport));
  memcpy(&sa.sin_addr, he->h_addr, 4);
  if ((sk = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
    die("socket: %m");
  if (connect(sk, (struct sockaddr *) &sa, sizeof(sa)) < 0)
    die("connect to %s:%d : %m", hostname, ntohs(sa.sin_port));

  if ((fd = sh_open(file, O_RDONLY, 0)) < 0)
    die("open(%s): %m", file);
  size = sh_seek(fd, 0, SEEK_END);
  if (trace)
    log(L_INFO, "Sending %s (%Ld bytes; compression %d)", file, (long long) size, compression);

  swrite(sk, &size, 8);
  swrite(sk, &compress_flag, 4);
  xfer_size = 0;
  if (compression)
    {
      z_stream zs;
      byte *outbuf = xmalloc(CHUNK_SIZE);
      byte *inbuf = NULL;
      int inchunk = 0;
      int mchunk = 0;
      int result;
      bzero(&zs, sizeof(zs));
      zs.data_type = Z_UNKNOWN;
      if (deflateInit(&zs, compression) != Z_OK)
	die("deflateInit failed: %s", zs.msg);
      do
	{
	  int cont = Z_NO_FLUSH;
	  if (!zs.avail_in)
	    {
	      if (size > 0)
		{
		  inchunk = CHUNK_SIZE;
		  if (inchunk > size)
		    inchunk = size;
		  mchunk = (inchunk + PAGE_SIZE - 1) & ~PAGE_SIZE;
		  if ((inbuf = sh_mmap(NULL, mchunk, PROT_READ, MAP_SHARED, fd, where)) == MAP_FAILED)
		    die("mmap: %m");
		  zs.next_in = inbuf;
		  zs.avail_in = inchunk;
		  size -= inchunk;
		}
	      else
		cont = Z_FINISH;
	    }
	  if (!zs.avail_out)
	    {
	      zs.next_out = outbuf;
	      zs.avail_out = CHUNK_SIZE;
	    }
	  result = deflate(&zs, cont);
	  if (result != Z_STREAM_END && result != Z_OK)
	    die("deflate error: %d (%s)", result, zs.msg);
	  if ((result == Z_STREAM_END || !zs.avail_out) && zs.next_out > outbuf)
	    {
	      u32 ws = zs.next_out-outbuf;
	      swrite(sk, &ws, sizeof(ws));
	      swrite(sk, outbuf, ws);
	      xfer_size += ws;
	    }
	  if ((result == Z_STREAM_END || !zs.avail_in) && mchunk)
	    {
	      munmap(inbuf, mchunk);
	      mchunk = 0;
	    }
	}
      while (result != Z_STREAM_END);
      adler = zs.adler;
      deflateEnd(&zs);
    }
  else
    {
      adler = adler32(0, NULL, 0);
      while (size > 0)
	{
	  void *buffer;
	  int chunk = CHUNK_SIZE;
	  uns mchunk;
	  if (chunk > size)
	    chunk = size;
	  mchunk = (chunk + PAGE_SIZE - 1) & ~(PAGE_SIZE-1);
	  if ((buffer = sh_mmap(NULL, mchunk, PROT_READ, MAP_SHARED, fd, where)) == MAP_FAILED)
	    die("mmap: %m");
	  swrite(sk, buffer, chunk);
	  adler = adler32(adler, buffer, chunk);
	  if (munmap(buffer, mchunk) < 0)
	    die("munmap: %m");
	  where += chunk;
	  size -= chunk;
	  xfer_size += chunk;
	}
    }

  MD5Init(&md5c);
  MD5Update(&md5c, key, strlen(key));
  MD5Update(&md5c, (byte *) &adler, sizeof(adler));
  MD5Final(md5, &md5c);
  swrite(sk, md5, sizeof(md5));

  close(fd);
  close(sk);

  time = get_timer();
  if (trace)
    log(L_INFO, "Transferred %Ld bytes in %.3f sec (%.3fMB/sec)",
	(long long) xfer_size,
	(double)time/1000,
	(double)xfer_size/1048576/((double)time/1000));
}

int main(int argc, char **argv)
{
  byte request[1024], *fields[4];
  byte *c;

  log_init("file-send");
  cf_register(filesend_config);
  if (cf_getopt(argc, argv, CF_SHORT_OPTS, CF_NO_LONG_OPTS, NULL) >= 0 ||
      optind+1 != argc)
    {
      fputs("Usage: file-send [<-C,-S as usually>] <directory>\n", stderr);
      return 1;
    }
  if (chdir(argv[argc-1]) < 0)
    die("chdir: %m");
  signal(SIGPIPE, SIG_IGN);

  while (fgets(request, sizeof(request), stdin))
    {
      c = strchr(request, '\n');
      if (!c)
	die("Request line too long");
      *c = 0;
      if (memcmp(request, "#FS# ", 5))
	continue;
      if (trace > 1)
	log(L_INFO, "<<< %s", request+5);
      if (wordsplit(request+5, fields, 4) != 4)
	die("Malformed request received");
      process_request(fields[0], fields[1], fields[2], fields[3]);
    }

  return 0;
}
