#!/bin/bash
#
#  Sherlock Search Server Control Script
#  (c) 2002--2003 Martin Mares <mj@ucw.cz>
#

set -e

if [ $UID = 0 ] ; then
	cd ${SH_HOME:-~sherlock/run}
	exec su ${SH_USER:-sherlock} -c "exec bin/scontrol $@"
else
	if [ -n "$SH_HOME" ] ; then cd $SH_HOME ; fi
fi
if ! [ -f bin/sherlockd -a -f cf/sherlock ] ; then
	echo >&2 "scontrol: Cannot find Sherlock runtime directory"
	exit 1
fi

eval `bin/config Search Port=8192 StatusFile '*'`
eval `bin/config SKeeper TestRetry=10 TestWait=3 CrashMail RotateLogs CrashWaitThreshold CrashWaitCeiling DaemonPIDFile KeeperPIDFile @TestQuery SwapLock SwapLockTimeout=10`

function log
{
	bin/logger scontrol I "$1"
}

function errlog
{
	bin/logger scontrol E "$1"
}

if [ -n "$CF_CrashWaitThreshold" ] ; then
	DHO="--pid-file=$CF_KeeperPIDFile"
	CMD=bin/skeeper
	MSG="sherlockd with skeeper"
else
	DHO="--pid-file=$CF_DaemonPIDFile"
	if [ -n "$CF_StatusFile" ] ; then
		DHO="$DHO --status-file=$CF_StatusFile"
	fi
	CMD=bin/sherlockd
	MSG="sherlockd"
fi

case "$1" in
	start)		echo -n "Starting $MSG..."
			# Bash is too weak for writing raceless startup/shutdown scripts,
			# so please forgive us a C helper.
			if bin/daemon-helper --check $DHO ; then
				echo " ALREADY RUNNING."
				exit 0
			fi
			bin/daemon-helper --start $DHO -- $CMD &
			PID=$!
			RETRY=$CF_TestRetry
			echo -n "."
			while true ; do
				if [ -n "$CF_StatusFile" -a ! -f $CF_StatusFile ] ; then
					# sherlockd isn't initialized yet, so no point in trying queries
					true
				elif bin/query --host localhost --port $CF_Port --silent --control '"databases"' >/dev/null 2>&1 ; then
					# sherlockd looks initialized, try sending a trivial query
					echo " done."
					exit 0
				fi
				if ! kill -0 $PID 2>/dev/null ; then
					# The daemon or daemon-helper has exited. Unfortunately, bash doesn't
					# give us any way how to get the return code, so in the rare case
					# sherlockd was already running and we haven't spotted it by --check,
					# we would report a failure incorrectly. Fortunately enough, if this
					# happens, the query check above exits with OK status first.
					echo " FAILED."
					exit 1
				fi
				if [ $RETRY = 0 ] ; then
					# The daemon got stuck, so go kill it
					echo " TIMED OUT."
					bin/scontrol stop
					exit 1
				fi
				RETRY=$(($RETRY-1))
				echo -n "."
				sleep $CF_TestWait
			done
			;;
	stop)		echo -n "Stopping $MSG... "
			if bin/daemon-helper --stop $DHO ; then
				if [ -n "$CF_CrashWaitThreshold" ] ; then
					# If we use the skeeper, kill skeeper first and then kill
					# the orphaned sherlockd, if there's any.
					bin/daemon-helper --stop --pid-file=$CF_DaemonPIDFile || true
				fi
				echo "done."
			elif [ $? == 102 ] ; then
				echo "not running."
			else
				echo "FAILED."
				exit 1
			fi
			;;
	restart)	bin/scontrol stop
			bin/scontrol start
			;;
	cron)		if [ -n "$CF_RotateLogs" ] ; then eval $CF_RotateLogs ; fi
			;;
	test)		[ -z "$CF_TestQuery" ] && exit 0
			echo -n "Testing sherlockd..."
			for test in "${CF_TestQuery[@]}" ; do
				echo -n "."
				MINREPLIES=`echo $test | cut -d ' ' -f 1`
				QUERY=`echo $test | cut -d ' ' -f 2-`
				T=tmp/test-$$
				if ! bin/query --host localhost --port $CF_Port --noheader --nofooter "$QUERY" >$T ; then
					rm $T
					echo "FAILED."
					errlog "Query '$QUERY' failed."
					exit 1
				fi
				c=`grep -c "^B" $T || true`
				rm $T
				if [ $c -lt $MINREPLIES ] ; then
					echo "FAILED."
					errlog "Query '$QUERY' produced only $c replies ($MINREPLIES required)."
					exit 1
				fi
			done
			echo " done."
			;;
	swap)		INDEX=${2:-index}
			log "Swapping $INDEX"
			if [ ! -f $INDEX.new/parameters ] ; then
				log "Nothing to swap"
				exit 0
			fi
			echo -n "Acquiring swap lock..."
			RETRY=$CF_SwapLockTimeout
			while ! mkdir $CF_SwapLock 2>/dev/null ; do
				if [ $RETRY = 0 ] ; then
					echo " TIMED OUT."
					exit 1
				fi
				RETRY=$(($RETRY-1))
				echo -n "."
				sleep 1
			done
			echo " done."
			bin/scontrol stop
			rm -rf $INDEX.old
			if [ -f $INDEX/parameters ] ; then
				mv $INDEX $INDEX.old
			else
				rm -rf $INDEX
			fi
			mv $INDEX.new $INDEX
			if bin/scontrol start ; then
				if bin/scontrol test ; then
					rmdir $CF_SwapLock
					exit 0
				fi
				bin/scontrol stop
			fi
			if [ -d $INDEX.old ] ; then
				log "Falling back to old $INDEX"
				mv $INDEX $INDEX.new
				mv $INDEX.old $INDEX
				if bin/scontrol start ; then
					log "Fallback succeeded"
				else
					errlog "Fallback failed"
				fi
			else
				errlog "No old index to fall back to"
			fi
			rmdir $CF_SwapLock
			exit 1
			;;
	*)		echo >&2 "Usage: [SH_USER=<user>] [SH_HOME=<homedir>] scontrol (start|stop|restart|cron|test|swap [<index>])"
			exit 1
			;;
esac
