#!/usr/bin/perl -w

# 
# Dr. Watson -- user-friendly frontend to statistics
#
# (c) 2003-2004 Tomas Valla <tom@ucw.cz>
#

# How to create custom pages with graphs:
#
# As ordinary CGI script you can supply arguments to drwatson.cgi.
# Possible arguments:
#  gath_stats 
#  shep_stats
#  search_stats 
#  idx_stats 		- write graph name into apropriate argument
#  search_pref 		- search server interfile prefixes, so what server you want to use
#  gath_all
#  exp_all 
#  idx_all 
#  search_all 		- set to nonempty if you want to view all graphs
#  bdate 
#  btime 
#  edate 
#  etime 		- date and time as in user defined time
#  force 		- force recomputing of graphs
#  time_input 		- should be one of lastday,lastweek,lastmonth,user
#  hide_office		- set to nonempty if you want to hide graph choosing screen
#
# Example:
# I'd like to view statistics of cache efficiency of sherlock0 and gatherer performance
# and spectrum for last week.
#
# drwatson.cgi?hide_office=true&time_input=lastweek&gath_stats=gatherd-performance&gath_stats=gatherd-spectrum&search_pref=sherlock0/sherlockd-&search_stats=search-cache_efficiency
#
#

use lib 'lib/perl5';
use Sherlock::CGI;
use Sherlock::Ulimit;
use POSIX;
use strict;
use warnings;

# you may wish to reconfigure the following stuff
my $graph_cgi_url = 'graph.cgi';

my $gth_stat_mask = "graph/gatherd-*";
my $srch_stat_mask = "graph/search-*";
my $srch_pref_mask = "log/inter/sherlock[0-9]*";
my $idx_stat_mask = "graph/indexer-*";

my $gatherd_prefix = "gather-";
my $indexer_prefix = "indexer-";

my @gath_allstats = map { (reverse split "/",$_)[0] } glob($gth_stat_mask);
my @shep_allstats= map { (reverse split "/",$_)[0] } glob($shep_stat_mask);
my @search_allstats = map { (reverse split "/",$_)[0] } glob($srch_stat_mask);
my @search_prefixes = map { (reverse split "/",$_)[0]."/sherlockd-" } glob($srch_pref_mask);
my @indexer_allstats = map { (reverse split "/",$_)[0] } glob($idx_stat_mask);

####

Sherlock::Ulimit::setlimit($Sherlock::Ulimit::AS, 40000000,40000000);

header("Dr. Watson");
body();
footer();

sub print_stats {
	my $stat_var = shift;
	my @r = ();

	foreach my $s (@_) {
		push @r, qq(<INPUT type="checkbox" name="$stat_var" value="$s">$s);
	}
	return @r;
}

sub header {
	my $title = shift;

	print <<END
Content-Type: text/html

<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">
<HTML>
<HEAD>
<TITLE>$title</TITLE>
</HEAD>
<BODY>
END
	;
}

sub footer {
	print "</BODY>\n</HTML>\n";
}

sub body {

	my ($bdate, $btime, $edate, $etime, $force, $time_input, $gath_all, $shep_all, $search_all, $idx_all, $hide_office);
	my (@gath_stats, @shep_stats, @search_stats, @search_pref, @idx_stats);
	my %pars = (
		gath_stats => {var => \@gath_stats},
		shep_stats => {var => \@shep_stats},
		search_stats => {var => \@search_stats},
		search_pref => {var => \@search_pref},
		idx_stats => {var => \@idx_stats},
		gath_all => {var => \$gath_all},
		shep_all => {var => \$shep_all},
		idx_all => {var => \$idx_all},
		search_all => {var => \$search_all},
		bdate => {var => \$bdate, check => '\d\d\d\d-\d{1,2}-\d{1,2}', default => '1990-01-01'},
		btime => {var => \$btime, check => '\d{1,2}:\d{1,2}', default => '00:00' },
		edate => {var => \$edate, check => '\d\d\d\d-\d{1,2}-\d{1,2}', default => strftime("%Y-%m-%d",localtime(time())) },
		etime => {var => \$etime, check => '\d{1,2}:\d{1,2}', default => '23:59' },
		force => {var => \$force },
		time_input => {var => \$time_input },
		hide_office => {var => \$hide_office },
	);
	Sherlock::CGI::parse_args(\%pars);

	if (! $hide_office) {
		print <<END
<H1>Dr. Watson's office</H1>
<PRE style="float: right">
   \\\\\\\\
   c  oo
    | .U
   __=__                        ,,,
  |.  __|___                    oo ;
  ||_/  /  /                    U= _  0
  \\_/__/__E   o                 /. .| |
   (___ ||    |~~~~~~~~~~~~~~~~'----'~|
   I---|||    |-----------------------|
   I   |||    |       c(__)           |
   ^   '--''  ^                       ^
</PRE>
<FORM method="post" action="?" enctype="application/x-www-form-urlencoded">
<H2>Give me a term</H2>
<INPUT type="radio" name="time_input" value="lastday" checked>Last day
<INPUT type="radio" name="time_input" value="lastweek">Last week
<INPUT type="radio" name="time_input" value="lastmonth">Last month
<BR>
<INPUT type="radio" name="time_input" value="user">User defined:
<BR>
<INPUT type="text" name="bdate" value="$bdate">
<INPUT type="text" name="btime" value="$btime">
Start date and time YYYY-MM-DD hh:mm<BR>

<INPUT type="text" name="edate" value="$edate">
<INPUT type="text" name="etime" value="$etime">
End date and time<BR>

<H2>What kind of examination would you like?</H2>
<TABLE border="1" cellspacing="0" cellpadding="3">
<CAPTION>Possible examinations</CAPTION>
<TR><TH>Gatherer<TH>Shepherd<TH>Search<TH>Indexer
<TR><TD><INPUT type="checkbox" name="gath_all" value="x">All<TD><INPUT type="checkbox" name="shep_all" value="x">All<TD><INPUT type="checkbox" name="search_all" value="x">All<TD><INPUT type="checkbox" name="idx_all" value="x">All
END
		;
		my @gs = reverse print_stats("gath_stats",@gath_allstats);
		my @hs = reverse print_stats("shep_stats",@shep_allstats);
		my @is = reverse print_stats("idx_stats",@indexer_allstats);
		my @ss = reverse print_stats("search_stats",@search_allstats);

		while (scalar(@gs+@is+@hs+@ss)>0) {
			my $g = pop @gs;
			my $h = pop @hs;
			my $i = pop @is;
			my $s = pop @ss;
			print "<TR><TD>".(defined $g ? $g : "")."<TD>".(defined $h ? $h : "")
				."<TD>".(defined $s ? $s : "")."<TD>".(defined $i ? $i : "")."\n";
		}
		print "</TABLE>\n";

		print "Search statistics for servers:\n";
		for my $p (@search_prefixes) {
			print qq(<INPUT type="checkbox" name="search_pref" value="$p");
			my @fn = glob("log/inter/$p*");
			if ((scalar @fn) == 0) { print ">$p\n" }
			else { print " checked>$p\n"} 
		}

		print <<END
<BR><INPUT type="checkbox" name="force" value="force">Force recomputing<BR>
<HR>
<INPUT type="submit" name="action" value="Examine">
<INPUT type="reset">
</FORM>
<P><A href="log/words/">See word statistics</A>
<HR>
END
		;
	}

	if ($time_input eq "lastday") {
		my $t = time();
		$bdate = strftime("%Y-%m-%d",localtime($t-24*3600));
		$btime = strftime("%H-%M",localtime($t-24*3600));
		$edate = strftime("%Y-%m-%d",localtime($t));
		$etime = strftime("%H-%M",localtime($t));
	} elsif ($time_input eq "lastweek") {
		my $t = time();
		$bdate = strftime("%Y-%m-%d",localtime($t-7*24*3600));
		$btime = strftime("%H-%M",localtime($t-7*24*3600));
		$edate = strftime("%Y-%m-%d",localtime($t));
		$etime = strftime("%H-%M",localtime($t));
	} elsif ($time_input eq "lastmonth") {
		my $t = time();
		$bdate = strftime("%Y-%m-%d",localtime($t-30*24*3600));
		$btime = strftime("%H-%M",localtime($t-30*24*3600));
		$edate = strftime("%Y-%m-%d",localtime($t));
		$etime = strftime("%H-%M",localtime($t));
	} else {
		$btime =~ s/:/-/;
		$etime =~ s/:/-/;
	}

	@gath_stats = @gath_allstats if $gath_all;
	@shep_stats  = @shep_allstats if $shep_all;
	@search_stats = @search_allstats if $search_all;
	@idx_stats = @indexer_allstats if $idx_all;

	print "<H3>Gatherer statistics</H3>\n" if @gath_stats>0;
	foreach my $g (@gath_stats) {
		my $url="$graph_cgi_url?name=$g&prefix=$gatherd_prefix&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
			.($force ? "&force=true" : "");
		print qq(<IMG src="$url" alt="There should be graph $g"><BR>\n);
	}

	print "<H3>Indexer statistics</H3>\n" if @idx_stats>0;
	foreach my $i (@idx_stats) {
		my $url="$graph_cgi_url?name=$i&prefix=$indexer_prefix&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
			.($force ? "&force=true" : "");
		print qq(<IMG src="$url" alt="There should be graph $i"><BR>\n);
	}

	foreach my $sp (@search_pref) {
		print "<H3>Search statistics for $sp</H3>\n" if @search_stats>0;
		foreach my $s (@search_stats) {
			my $url="$graph_cgi_url?name=$s&prefix=$sp&bdate=$bdate&btime=$btime&edate=$edate&etime=$etime"
				.($force ? "&force=true" : "");
			print qq(<IMG src="$url" alt="There should be graph $s"><BR>\n);
		}
	}
}

