/*
 *	Resolver for fingerprints
 *
 *	(c) 2003--2004, Robert Spalek <robert@ucw.cz>
 */

#undef LOCAL_DEBUG

#include "sherlock/sherlock.h"
#include "lib/fastbuf.h"
#include "indexer/indexer.h"

#include <stdlib.h>
#include <fcntl.h>

sh_off_t
resolve_optimize_run_length(struct fastbuf *in UNUSED)
{
  struct fastbuf *tmp = index_bopen(fn_fingerprints, O_RDONLY);
  sh_off_t len_fp = bfilesize(tmp);
  bclose(tmp);
  /*
   * The following solution minimizes the total amount of read/written data:
   * MIN_rl ( 2 len_in * log (rl / sort_buffer) + len_in / rl * len_fp + links )
   */
  sh_off_t rl = len_fp / 2;
  ITRACE("Average run length needs to be >= %dM", (uns) (rl >> 20));
  return rl;
}

static sh_off_t fpsort_bs;

#define SORT_KEY	struct card_print
#define SORT_PREFIX(x)	ri_##x
#define SORT_PRESORT
#define SORT_REGULAR
#define SORT_INPUT_FB
#define SORT_OUTPUT_FB
#define SORT_UP_TO	fpsort_bs

static inline int
ri_compare(struct card_print *a, struct card_print *b)
{
  return memcmp(&a->fp, &b->fp, sizeof(struct fingerprint));
}

#include "lib/sorter.h"

#define SORT_KEY	struct resolve_output
#define SORT_PREFIX(x)	ro_##x
#define SORT_PRESORT
#define SORT_REGULAR
#define SORT_INPUT_FB
#define SORT_OUTPUT_FB

static uns ro_mask;

static inline int
ro_compare(struct resolve_output *a, struct resolve_output *b)
{
  uns ai = a->src & ro_mask;
  uns bi = b->src & ro_mask;
  if (ai < bi)
    return -1;
  else if (ai > bi)
    return 1;
  else
    return 0;
}

#include "lib/sorter.h"

struct fastbuf *
resolve_fastbuf(struct fastbuf *in, uns flags, uns add_size)
{
  /* Merge sorted in with sorted fingerprints, close in, and create out.
   * Records consist of struct card_print and add_size additional bytes.  */
  struct card_print resolve, sorted;
  byte add[add_size];
  struct fastbuf *out = bopen_tmp(indexer_fb_size);
  struct fastbuf *fps = index_bopen(fn_fingerprints, O_RDONLY);
  bread(fps, &sorted, sizeof(struct card_print)) || (sorted.cardid = ~0U);
  uns resolved = 0, resolved_new = 0, unknown = 0;
  struct fingerprint last;
  uns seeks = 0;
  bzero(&last, sizeof(struct fingerprint));
  while (bread(in, &resolve, sizeof(struct card_print)))
  {
    int cmp = memcmp(&resolve.fp, &last, sizeof(struct fingerprint));
    if (cmp < 0)
    {
      bsetpos(fps, 0);
      bread(fps, &sorted, sizeof(struct card_print)) || (sorted.cardid = ~0U);
      seeks++;
      DBG("Resolve: seeking back");
    }
    if (cmp)
      memcpy(&last, &resolve.fp, sizeof(struct fingerprint));
    if (add_size)
      bread(in, add, add_size);
    struct resolve_output ro;
    int res;
    if (sorted.cardid == ~0U)
      res = -1;
    else
      res = memcmp(&resolve.fp, &sorted.fp, sizeof(struct fingerprint));
    ro.src = resolve.cardid;
again:
    if (res < 0)
    {
      if (!(flags & RESOLVE_SKIP_UNKNOWN))
      {
	ro.dest = ~0U;
	bwrite(out, &ro, sizeof(ro));
	if (add_size)
	  bwrite(out, add, add_size);
      }
      unknown++;
    }
    else if (!res)
    {
      if (sorted.cardid < FIRST_ID_NEW || !(flags & RESOLVE_SKIP_NEW))
      {
	ro.dest = sorted.cardid;
	bwrite(out, &ro, sizeof(ro));
	if (add_size)
	  bwrite(out, add, add_size);
      }
      if (sorted.cardid >= FIRST_ID_NEW)
	resolved_new++;
      else
	resolved++;
    }
    else
    {
      do
      {
	if (!bread(fps, &sorted, sizeof(struct card_print)))
	{
	  sorted.cardid = ~0U;
	  res = -1;
	  break;
	}
	res = memcmp(&resolve.fp, &sorted.fp, sizeof(struct fingerprint));
      }
      while (res > 0);
      goto again;
    }
  }
  bclose(fps);
  bclose(in);
  brewind(out);
  log(L_INFO, "Resolve with %u add_size: %u + %u new resolved, %u unknown; %u seeks", add_size, resolved, resolved_new, unknown, seeks);
  return out;
}

struct fastbuf *
resolve_fingerprints(struct fastbuf *in, uns flags, uns src_mask)
{
  fpsort_bs = resolve_optimize_run_length(in);
  log(L_INFO, "Resolve: sorting by fingerprint");
  in = ri_sort(in);
  log(L_INFO, "Resolving the fingerprints");
  struct fastbuf *out = resolve_fastbuf(in, flags, 0);
  log(L_INFO, "Resolve: sorting by cardid");
  ro_mask = src_mask;
  return ro_sort(out);
}
