#!/usr/bin/perl -w
#
# Joins gatherer errors with reapd error messages and show mux errors
#
# (c) 2005 Vladimir Jelen <vladimir.jelen@netcentrum.cz>
#
# usage: bin/log-gatherer-bugs <gatherer-log-files> <mux-log-files>

use strict;
use Time::Local;
use Getopt::Long;
use File::stat;

my %errors = ();
my @finput = @ARGV;
my @rinput = ();
my @goutput = ();
my @moutput = ();

my $gserver = '';
my $mserver = '';
my %rserver = ();

my $listfile = '';
my @analyzed = ();
my $force = 0;
my $now = time;

# Support for opening compressed files
sub open_method ($) {
  $_ = shift;
  my $retval;
  if(/\.gz$/) { $retval = "zcat $_ |"; }
  elsif(/\.bz2$/) { $retval = "bzcat $_ |"; }
  else { $retval = $_; }
  return $retval;
}

# Function for reading watson config values
sub read_config_value($$) {
  my ($var,$val) = @_;
  my $tmp = `. cf/watson && echo -n \$$val`;
  if($tmp ne '') { $$var = $tmp; }
}

GetOptions("force!" => \$force);

# Read config
read_config_value(\$listfile, 'LOG_LIST');
read_config_value(\$gserver, 'GATHERD_PREFIX');
read_config_value(\$mserver, 'MUX_PREFIX');
read_config_value(\$_, 'REAP_PREFIX');

foreach my $rs (split) {
  $rs =~ /^([^:]+):(.+)$/ && do {
    $rserver{"$1"} = $2;
  }
}

# Read list of analyzed logs
open(LISTFILE, $listfile);
while(<LISTFILE>) {
  chomp;
  push @analyzed, $_;
}
close LISTFILE;
open(LISTFILE, '>>', $listfile) or die "Cannot open file $listfile: $!";

# Analyze gatherer logs
ANALYZE: foreach my $logfile (@finput) {
  my $norm = $logfile;
  if($logfile =~ /^(.+)(\.(gz|bz2))$/) { $norm = $1; }
  -f $logfile or next ANALYZE;
  if(!$force) {
    my $tmp = stat $logfile or next ANALYZE;
    if($tmp->mtime > ($now - 86400)) { next ANALYZE; }
    foreach $_ (@analyzed) {
      $norm eq $_ && next ANALYZE;
    }
  }
  if ($logfile =~ /^$gserver(\d{4})(\d{2})(\d{2})(\.(gz|bz2))?$/) {
    foreach $_ (values %rserver) { push @rinput, "$_$1$2$3"; }
    open(LOGFILE, open_method($logfile)) or die "Cannot open file $logfile: $!";
    while(<LOGFILE>) {
      chomp;
      if(/^I (\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2}) \S+ [12]301 Gatherer bug \[([^:]+):([^*\]]+)/) {
        my $date = timegm($6, $5, $4, $3, $2 - 1, $1);
        $errors{$8} = [] if not defined $errors{$8};
        push @{$errors{$8}}, { 'id' => $8, 'rserv' => $7, 'date' => $date, 'txt' => $_ };
      }
    }
  }
  elsif ($logfile =~ /^$mserver(\d{4})(\d{2})(\d{2})(\.(gz|bz2))?$/) {
    open(LOGFILE, open_method($logfile)) or die "Cannot open file $logfile: $!";
    while(<LOGFILE>) {
      chomp;
      if (/^!/ || /^E/) { push @moutput, $_; }
    }
  }
  if(!$force) { print LISTFILE "$norm\n"; }
}

# Analyze & join reapd logs
foreach my $logfile (@rinput) {
  if($logfile =~ /reapd/i) {
    open(LOGFILE, open_method($logfile)) or open(LOGFILE, open_method($logfile . '.gz')) or open(LOGFILE, open_method($logfile . '.bz2')) or die "Cannot open file $logfile: $!";
    while(<LOGFILE>) {
      chomp;
      if(/^. (\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2}) \[0+([0-9a-fA-F]+)/) {
        if(defined $errors{$7}) {
          my $date = timegm($6, $5, $4, $3, $2 - 1, $1);
          foreach my $gath (@{$errors{$7}}) {
            my $rs = $rserver{$$gath{'rserv'}};
            next if abs($$gath{'date'} - $date) > 120;
            next if not $logfile =~ /^$rs/;
            $$gath{'reap'} = [] if not defined $$gath{'reap'};
            push @{$$gath{'reap'}}, $_;
          }
        }
      }
    }
  }
}

#Group results
foreach my $data (values %errors) {
  push @goutput, @$data;
}

# Print results
if(@goutput > 0) {
    print "Gatherer errors\n" . "-" x 80 . "\n\n";
    foreach $_ (sort { $$a{'date'} <=> $$b{'date'} } @goutput) {
    print $$_{'txt'} . "\n";
    foreach my $reap_txt (@{$$_{'reap'}}) {
      print $reap_txt . "\n";
    }
    print "\n";
  }
  print "\n\n";
}

if(@moutput > 0) {
  print "Mux errors\n" . "-" x 80 . "\n\n";
  $" = "\n";
  print "@moutput\n";
}
